import datetime
import os
import random

import numpy as np
import pytz
import torch
from transformers import PreTrainedModel, PreTrainedTokenizer


def debug_model(
        _model: PreTrainedModel,
        _tokenizer: PreTrainedTokenizer,
        device: torch.device = torch.device('cpu')
):
    """
    Inspect the model by its perplexity
    """
    _test_sentences = [
        "Shall I compare thee to a summer's day?",
        "Though art more lovely and more temperate.",
        "Rough winds do shake the darling buds of May,",
        "And summer's lease hath all too short a date."
    ]
    _test_inputs = _tokenizer(_test_sentences, padding=True,
                              truncation=True, return_tensors='pt', max_length=128)
    _test_inputs['labels'] = _test_inputs['input_ids'].clone()
    _model.eval()
    _test_outputs = _model(input_ids=_test_inputs['input_ids'],
                           attention_mask=_test_inputs['attention_mask'],
                           labels=_test_inputs['labels'])
    print(f"Inspect the model with {_model.num_parameters()} parameters "
          f"by perplexity: {torch.exp(_test_outputs.loss)}")


def seed_all(seed: int = 42):
    """
    Set all seeds to reproduce the results
    """
    np.random.seed(seed)
    random.seed(seed)
    torch.manual_seed(seed)
    os.environ['PYTHONHASHSEED'] = str(seed)
    if torch.cuda.is_available():
        torch.cuda.manual_seed_all(seed)


def experiment_name_time(run_name: str) -> str:
    """
    Generate a unique experiment name with current time

    Examples
    --------
    >>> experiment_name_time("gpt4") # doctest: +SKIP
    gpt4-2023.6.18.21.2.31'
    """
    current_time = datetime.datetime.now(pytz.timezone('Asia/Shanghai'))
    exp_name = run_name + "-" + "{0}.{1}.{2}.{3}.{4}.{5}".format(
        current_time.year,
        current_time.month,
        current_time.day,
        current_time.hour,
        current_time.minute,
        current_time.second,
    )
    return exp_name
