import argparse
import torch
from llava.auto_vision_prompt.train_obj_detector import make_supervised_data_module_classification, ClassificationSupervisedDataset, init_vision_tower_delayed

from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, get_model_name_from_path, KeywordsStoppingCriteria

from PIL import Image

import requests
from PIL import Image
from io import BytesIO


def load_image(image_file):
    if image_file.startswith('http') or image_file.startswith('https'):
        response = requests.get(image_file)
        image = Image.open(BytesIO(response.content)).convert('RGB')
    else:
        image = Image.open(image_file).convert('RGB')
    return image


def eval_model(args):
    # Model
    disable_torch_init()

    model_name = get_model_name_from_path(args.model_path)
    tokenizer, model, image_processor, context_len = load_pretrained_model(args.model_path, args.model_base, model_name)

    qs = args.query
    if model.config.mm_use_im_start_end:
        qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
    else:
        qs = DEFAULT_IMAGE_TOKEN + '\n' + qs

    if 'llama-2' in model_name.lower():
        conv_mode = "llava_llama_2"
    elif "v1" in model_name.lower():
        conv_mode = "llava_v1"
    elif "mpt" in model_name.lower():
        conv_mode = "mpt"
    else:
        conv_mode = "llava_v0"


        

    if args.conv_mode is not None and conv_mode != args.conv_mode:
        print('[WARNING] the auto inferred conversation mode is {}, while `--conv-mode` is {}, using {}'.format(conv_mode, args.conv_mode, args.conv_mode))
    else:
        args.conv_mode = conv_mode

    conv = conv_templates[args.conv_mode].copy()
    conv.append_message(conv.roles[0], qs)
    conv.append_message(conv.roles[1], None)
    prompt = conv.get_prompt()

    image = load_image(args.image_file)
    image_tensor = image_processor.preprocess(image, return_tensors='pt')['pixel_values'].half().cuda()

    input_ids = tokenizer_image_token(prompt, tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()

    stop_str = conv.sep if conv.sep_style != SeparatorStyle.TWO else conv.sep2
    keywords = [stop_str]
    stopping_criteria = KeywordsStoppingCriteria(keywords, tokenizer, input_ids)

    with torch.inference_mode():
        output_ids = model.generate(
            input_ids,
            images=image_tensor,
            do_sample=True,
            temperature=0.2,
            max_new_tokens=1024,
            use_cache=True,
            stopping_criteria=[stopping_criteria])

    input_token_len = input_ids.shape[1]
    n_diff_input_output = (input_ids != output_ids[:, :input_token_len]).sum().item()
    if n_diff_input_output > 0:
        print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
    outputs = tokenizer.batch_decode(output_ids[:, input_token_len:], skip_special_tokens=True)[0]
    outputs = outputs.strip()
    if outputs.endswith(stop_str):
        outputs = outputs[:-len(stop_str)]
    outputs = outputs.strip()
    print(outputs)

if __name__ == "__main__":
    # parser = argparse.ArgumentParser()
    # parser.add_argument("--model_path", type=str, default="./checkpoints/llava-llava-llama-2-7b-chat-lightning-preview-obj")
    # parser.add_argument("--data_path", type=str, required=True)
    # parser.add_argument("--query", type=str, required=True)
    # parser.add_argument("--conv-mode", type=str, default=None)
    # args = parser.parse_args()

    # eval_model(args)
    import transformers
    from llava.model.language_model.llava_llama import LlavaVisionForCausalLM
    
    from llava.auto_vision_prompt.train_obj_detector import ModelArguments, DataArguments, TrainingArguments
    
    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments))
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()

    model_name_or_path = "./checkpoints/llava-llava-llama-2-7b-chat-lightning-preview-obj-4"
    config = transformers.AutoConfig.from_pretrained("./checkpoints/llava-llama-2-7b-chat-lightning-preview", trust_remote_code=True)
    model = LlavaVisionForCausalLM(config=config, output_dir=model_name_or_path)

    compute_dtype = (torch.float16 if training_args.fp16 else (torch.bfloat16 if training_args.bf16 else torch.float32))
    tokenizer = transformers.AutoTokenizer.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=training_args.cache_dir,
            model_max_length=training_args.model_max_length,
            padding_side="right",
            use_fast=False,
        )
    model = init_vision_tower_delayed(model, model_args, data_args, training_args, tokenizer, compute_dtype)

    module = make_supervised_data_module_classification(data_args)

    train_dataset = ClassificationSupervisedDataset(label_dict_path=data_args.data_path, image_folder=data_args.image_folder,
                            data_args=data_args)
    
    for i in range(0, len(train_dataset), 6):
        data = train_dataset[i]
        images, labels = data['images'], data['labels']
        print(f"images shape: {images.shape}")
        print(f"labels shape: {labels.shape}")
        output = model.inference(labels, images)
        print(f"output keys{output.keys()}")
        print(f"output logits shape: {output['logits'].shape}")
        print(f"output loss: {output['loss']}")
        print(f"output predict_classes: {output['predict_classes']}")
        print(f"output inputs_embeds: {output['inputs_embeds'].shape}")

    
