import os
from PIL import Image
import json
from tqdm import tqdm
import shortuuid

import torch
import transformers
from transformers import LogitsProcessorList, TemperatureLogitsWarper, TopPLogitsWarper

from llava.eval.utils_cfg import CFGLogits
from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, get_model_name_from_path, KeywordsStoppingCriteria
from llava.training_module.load_args import ModelArguments, DataArguments, TrainingArguments, OtherArguments
from llava.training_module.utils import get_chunk, update_mm_projector, get_answers_file_name
from llava.model.builder import load_pretrained_model_v2, load_pretrained_vision_module

def eval_model():
    ## load args
    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments, OtherArguments))
    model_args, data_args, training_args, args = parser.parse_args_into_dataclasses()
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    
    disable_torch_init()

    ## load vision module
    if args.cfg_mode == 'vision':
        visual_tokenizer, visual_model, image_processor, context_len = load_pretrained_vision_module(load_from_ckpt=True, model_args=model_args, data_args=data_args, training_args=training_args, model_name="LlavaVision", device=device)
    
    ## load LLAVA model
    model_base = None if args.model_base is None else args.model_base
    model_path = model_args.model_name_or_path if model_base is None else args.model_path
    model_name = get_model_name_from_path(model_path)
    tokenizer, model, image_processor, context_len = load_pretrained_model_v2(model_args, data_args, training_args, model_path, model_base, model_name)
    
    # ## initialize vision modules
    # model = init_vision_module(model, model_args, data_args, training_args, tokenizer)

    ## update mm projector
    if args.tune_manual_mm_projector_path is not None:
        # need to init vision module first, then load the mm projector
        model = update_mm_projector(model, args.tune_manual_mm_projector_path)

    # check trainable parameters
    from llava.training_module.utils import print_trainable_layers, print_model_size
    print_trainable_layers(model)

    ## load data
    questions = json.load(open(os.path.expanduser(os.path.join(args.question_path, args.question_file)), "r"))
    questions = get_chunk(questions, args.num_chunks, args.chunk_idx)
    
    if args.answers_file is None: 
        args.answers_file = get_answers_file_name(args, model_name, model_args.pretrain_mm_mlp_adapter)
       
    answers_file = os.path.expanduser(os.path.join(args.answer_path, args.answers_file))
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    ans_file = open(answers_file, "w")

    for i, line in enumerate(tqdm(questions)):
        idx = line["id"]
        question = line['conversations'][0]
        gt_ans = line["conversations"][1]['value']
        qs = question['value'].replace('<image>', '').strip()
        cur_prompt = qs

        if 'image' in line:
            image_file = line["image"]
            image = Image.open(os.path.join(args.eval_image_folder, image_file))
            image_tensor = image_processor.preprocess(image, return_tensors='pt')['pixel_values'][0]
            images = image_tensor.unsqueeze(0).half().cuda()
            if getattr(model.config, 'mm_use_im_start_end', False):
                qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
            else:
                qs = DEFAULT_IMAGE_TOKEN + '\n' + qs
            cur_prompt = '<image>' + '\n' + cur_prompt
        else:
            images = None

        if args.cfg_mode == 'text':
            neg_prompt = tokenizer(gt_ans, return_tensors='pt')['input_ids'].to(device)
        elif args.cfg_mode == 'vision':
            with torch.inference_mode():
                visual_output = visual_model.generate_vision_prompt(images)
                neg_prompt = visual_output.to(device)
        else:
            raise ValueError(f"cfg_mode {args.cfg_mode} is not supported!")

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        input_ids = tokenizer_image_token(prompt, tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()

        stop_str = conv.sep if conv.sep_style != SeparatorStyle.TWO else conv.sep2
        keywords = [stop_str]
        stopping_criteria = KeywordsStoppingCriteria(keywords, tokenizer, input_ids)

        ## inference
        with torch.inference_mode():
            if args.cfg == 0:
                output_ids = model.generate(
                    input_ids,
                    images=images,
                    do_sample=True,
                    temperature=0.2,
                    max_new_tokens=1024,
                    use_cache=True,
                    stopping_criteria=[stopping_criteria],
                )
            else:
                output_ids = model.generate(
                    input_ids,
                    images=images,
                    do_sample=True,
                    temperature=0.2,
                    max_new_tokens=1024,
                    use_cache=True,
                    stopping_criteria=[stopping_criteria],
                    logits_processor=LogitsProcessorList([
                        # inputs_cfg usually is the last token of the prompt but there are
                        # possibilities of negative prompting that are explored in the paper
                        CFGLogits(args.cfg, neg_prompt, model, cfg_mode=args.cfg_mode),
                        TemperatureLogitsWarper(0.8),
                        TopPLogitsWarper(0.95),
            ]),)

        input_token_len = input_ids.shape[1]
        n_diff_input_output = (input_ids != output_ids[:, :input_token_len]).sum().item()
        if n_diff_input_output > 0:
            print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
        outputs = tokenizer.batch_decode(output_ids[:, input_token_len:], skip_special_tokens=True)[0]
        outputs = outputs.strip()
        if outputs.endswith(stop_str):
            outputs = outputs[:-len(stop_str)]
        outputs = outputs.strip()

        ## prompt for answer
        if args.answer_prompter:
            outputs_reasoning = outputs
            input_ids = tokenizer_image_token(prompt + outputs_reasoning + ' ###\nANSWER:', tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()

            with torch.inference_mode():
                if args.cfg == 0:
                    output_ids = model.generate(
                        input_ids,
                        images=images,
                        do_sample=True,
                        temperature=0.2,
                        max_new_tokens=1024,
                        use_cache=True,
                        stopping_criteria=[stopping_criteria],
                    )
                else:
                    output_ids = model.generate(
                        input_ids,
                        images=images,
                        do_sample=True,
                        temperature=0.2,
                        max_new_tokens=1024,
                        use_cache=True,
                        stopping_criteria=[stopping_criteria],
                        logits_processor=LogitsProcessorList([
                            # inputs_cfg usually is the last token of the prompt but there are
                            # possibilities of negative prompting that are explored in the paper
                            CFGLogits(args.cfg, neg_prompt, model),
                            TemperatureLogitsWarper(0.8),
                            TopPLogitsWarper(0.95),
                ]),)

            input_token_len = input_ids.shape[1]
            n_diff_input_output = (input_ids != output_ids[:, :input_token_len]).sum().item()
            if n_diff_input_output > 0:
                print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
            outputs = tokenizer.batch_decode(output_ids[:, input_token_len:], skip_special_tokens=True)[0]
            outputs = outputs.strip()
            if outputs.endswith(stop_str):
                outputs = outputs[:-len(stop_str)]
            outputs = outputs.strip()
            outputs = outputs_reasoning + '\n The answer is ' + outputs

        ans_id = shortuuid.uuid()
        ans_file.write(json.dumps({"question_id": idx,
                                   "prompt": cur_prompt,
                                   "text": outputs,
                                   "answer_id": ans_id,
                                   "model_id": model_name,
                                   "metadata": {}}) + "\n")
        ans_file.flush()
    ans_file.close()
    print(f"saved answers to {answers_file}")

if __name__ == "__main__":
    eval_model()
