import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid
from PIL import Image
import math

import transformers
from transformers import LogitsProcessorList, TemperatureLogitsWarper, TopPLogitsWarper

from llava.eval.utils_cfg import CFGLogits
from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, get_model_name_from_path, KeywordsStoppingCriteria
from llava.auto_vision_prompt.train_obj_detector import load_pretrained_vision_module
from llava.training_module.load_args import ModelArguments, DataArguments, TrainingArguments



def split_list(lst, n):
    """Split a list into n (roughly) equal-sized chunks"""
    chunk_size = math.ceil(len(lst) / n)  # integer division
    return [lst[i:i+chunk_size] for i in range(0, len(lst), chunk_size)]


def get_chunk(lst, n, k):
    # convert dataframes to lists
    lst = list(lst)
    chunks = split_list(lst, n)
    return chunks[k]


def update_mm_projector(model, mm_projector_path):
    loaded_weights = torch.load(mm_projector_path, map_location="cpu")
    model.model.mm_projector.load_state_dict({"weight": loaded_weights["model.mm_projector.weight"], "bias": loaded_weights["model.mm_projector.bias"]})
    print("updated mm_projector weights!")
    return model


def eval_model(args):
    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments))
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    

    # Model
    disable_torch_init()

    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    if args.cfg_mode == 'image':
        visual_tokenizer, visual_model, image_processor, context_len = load_pretrained_vision_module(load_from_ckpt=True, model_args=model_args, data_args=data_args, training_args=training_args, model_name="LlavaVision", device=device)
    
    model_path = os.path.expanduser(args.model_path)
    model_name = get_model_name_from_path(model_path)

    tokenizer, model, image_processor, context_len = load_pretrained_model(model_path, args.model_base, model_name)

    if args.tune_manual_mm_projector_path is not None:
        model = update_mm_projector(model, args.tune_manual_mm_projector_path)

    questions = json.load(open(os.path.expanduser(os.path.join(args.question_path, args.question_file)), "r"))
    questions = get_chunk(questions, args.num_chunks, args.chunk_idx)
    
    if args.answers_file is None:
        if '7b' in model_path:
            args.answers_file = args.question_file.replace('.json', f'_7b.jsonl')
        else:
            args.answers_file = args.question_file.replace('.json', f'.jsonl')
        
        if args.tune_manual_mm_projector_path is not None:
            args.answers_file = args.answers_file.replace('.jsonl', f'_tune.jsonl')    
        if args.cfg_mode == 'image':
            args.answers_file = args.answers_file.replace('.jsonl', '_image.jsonl')

        args.answers_file = args.answers_file.replace('.jsonl', f'_cfg{args.cfg}.jsonl')
    answers_file = os.path.expanduser(os.path.join(args.answer_path, args.answers_file))
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    ans_file = open(answers_file, "w")


    for i, line in enumerate(tqdm(questions)):
        idx = line["id"]
        question = line['conversations'][0]
        gt_ans = line["conversations"][1]['value']
        qs = question['value'].replace('<image>', '').strip()
        cur_prompt = qs

        if 'image' in line:
            image_file = line["image"]
            image = Image.open(os.path.join(args.eval_image_folder, image_file))
            image_tensor = image_processor.preprocess(image, return_tensors='pt')['pixel_values'][0]
            images = image_tensor.unsqueeze(0).half().cuda()
            if getattr(model.config, 'mm_use_im_start_end', False):
                qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
            else:
                qs = DEFAULT_IMAGE_TOKEN + '\n' + qs
            cur_prompt = '<image>' + '\n' + cur_prompt
        else:
            images = None

        if args.cfg_mode == 'text':
            neg_prompt = tokenizer(gt_ans, return_tensors='pt')['input_ids'].to(device)
        elif args.cfg_mode == 'image':
            with torch.inference_mode():
                # print(f"shape of images: {images.shape} {images.dtype}") # shape of images: torch.Size([1, 3, 224, 224]) torch.float16
                # shape of neg_prompt: torch.Size([1, 256, 4096])
                visual_output = visual_model.generate_vision_prompt(images)
                # neg_prompt = visual_output['inputs_embeds'].to(device)
                neg_prompt = visual_output.to(device)
                # to float 32
                # neg_prompt = neg_prompt.to(torch.float32)
                 # TODO: check if this is correct
        else:
            raise ValueError(f"cfg_mode {args.cfg_mode} is not supported!")

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        input_ids = tokenizer_image_token(prompt, tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()

        stop_str = conv.sep if conv.sep_style != SeparatorStyle.TWO else conv.sep2
        keywords = [stop_str]
        stopping_criteria = KeywordsStoppingCriteria(keywords, tokenizer, input_ids)

        with torch.inference_mode():
            if args.cfg == 0:
                output_ids = model.generate(
                    input_ids,
                    images=images,
                    do_sample=True,
                    temperature=0.2,
                    max_new_tokens=1024,
                    use_cache=True,
                    stopping_criteria=[stopping_criteria],
                )
            else:
                output_ids = model.generate(
                    input_ids,
                    images=images,
                    do_sample=True,
                    temperature=0.2,
                    max_new_tokens=1024,
                    use_cache=True,
                    stopping_criteria=[stopping_criteria],
                    logits_processor=LogitsProcessorList([
                        # inputs_cfg usually is the last token of the prompt but there are
                        # possibilities of negative prompting that are explored in the paper
                        CFGLogits(args.cfg, neg_prompt, model, cfg_mode=args.cfg_mode),
                        TemperatureLogitsWarper(0.8),
                        TopPLogitsWarper(0.95),
            ]),)

        input_token_len = input_ids.shape[1]
        n_diff_input_output = (input_ids != output_ids[:, :input_token_len]).sum().item()
        if n_diff_input_output > 0:
            print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
        outputs = tokenizer.batch_decode(output_ids[:, input_token_len:], skip_special_tokens=True)[0]
        outputs = outputs.strip()
        if outputs.endswith(stop_str):
            outputs = outputs[:-len(stop_str)]
        outputs = outputs.strip()

        # prompt for answer
        if args.answer_prompter:
            outputs_reasoning = outputs
            input_ids = tokenizer_image_token(prompt + outputs_reasoning + ' ###\nANSWER:', tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()

            with torch.inference_mode():
                if args.cfg == 0:
                    output_ids = model.generate(
                        input_ids,
                        images=images,
                        do_sample=True,
                        temperature=0.2,
                        max_new_tokens=1024,
                        use_cache=True,
                        stopping_criteria=[stopping_criteria],
                    )
                else:
                    output_ids = model.generate(
                        input_ids,
                        images=images,
                        do_sample=True,
                        temperature=0.2,
                        max_new_tokens=1024,
                        use_cache=True,
                        stopping_criteria=[stopping_criteria],
                        logits_processor=LogitsProcessorList([
                            # inputs_cfg usually is the last token of the prompt but there are
                            # possibilities of negative prompting that are explored in the paper
                            CFGLogits(args.cfg, neg_prompt, model),
                            TemperatureLogitsWarper(0.8),
                            TopPLogitsWarper(0.95),
                ]),)

            input_token_len = input_ids.shape[1]
            n_diff_input_output = (input_ids != output_ids[:, :input_token_len]).sum().item()
            if n_diff_input_output > 0:
                print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
            outputs = tokenizer.batch_decode(output_ids[:, input_token_len:], skip_special_tokens=True)[0]
            outputs = outputs.strip()
            if outputs.endswith(stop_str):
                outputs = outputs[:-len(stop_str)]
            outputs = outputs.strip()
            outputs = outputs_reasoning + '\n The answer is ' + outputs

        ans_id = shortuuid.uuid()
        ans_file.write(json.dumps({"question_id": idx,
                                   "prompt": cur_prompt,
                                   "text": outputs,
                                   "answer_id": ans_id,
                                   "model_id": model_name,
                                   "metadata": {}}) + "\n")
        ans_file.flush()
    ans_file.close()
    print(f"Saved answers to {answers_file}")

if __name__ == "__main__":

    with open('./llava/eval/model_vqa_args.json') as json_file:
        args = json.load(json_file)
    args = argparse.Namespace(**args)

    eval_model(args)
