import multiprocessing
from typing import Any, Callable
from common.task import RegexTask
from common.verdict import SolverReply
from evaluators.solvers.regex.deepregex.backend import (
    regex_equiv
)


def _timeout_worker(conn, func: Callable, args: Any, kwargs: Any):
    try:
        res = func(*args, **kwargs)
        conn.send((True, res))
    except BaseException as e:
        conn.send((False, (e.__class__.__name__, str(e))))
    finally:
        try:
            conn.close()
        except Exception:
            pass


def run_with_timeout(func: Callable, timeout_ms: int, *args, **kwargs) -> Any:
    ctx = multiprocessing.get_context("spawn")
    parent_conn, child_conn = ctx.Pipe(duplex=False)

    p = ctx.Process(
        target=_timeout_worker,
        args=(child_conn, func, args, kwargs),
        daemon=False,
    )
    p.start()
    child_conn.close()

    try:
        if parent_conn.poll(timeout_ms / 1000.0):
            ok, payload = parent_conn.recv()
            p.join()
            if ok:
                return payload
            exc_type, exc_msg = payload
            raise RuntimeError(f"{exc_type}: {exc_msg}")
        else:
            p.terminate()
            p.join()
            raise TimeoutError()
    finally:
        try:
            parent_conn.close()
        except Exception:
            pass


def solve_regex(*, task: RegexTask, timeout_ms: int) -> SolverReply:
    llm_answer = task.llm_solution
    if llm_answer == "" or llm_answer is None:
        return SolverReply(verdict="failure", error_message="Empty LLM answer")
    llm_answer = llm_answer.replace(' ', '').replace('\n', '').replace('\t', '')
    try:
        is_equiv = run_with_timeout(
            regex_equiv, timeout_ms, task.answer, llm_answer
        )
        if is_equiv is True:
            return SolverReply(verdict="success")
        elif is_equiv is False:
            return SolverReply(verdict="failure")
        else:
            return SolverReply(verdict="unknown")
    except TimeoutError:
        return SolverReply(verdict="unknown", error_message="Timeout")
    except Exception as e:
        return SolverReply(
            verdict="unknown", error_message=f"Error during solving: {e}"
        )
