import re
from pathlib import Path
from dataclasses import dataclass


@dataclass
class CoqFile:
    filepath: Path
    full_content: str
    prompt: str | None = None
    imports: str | None = None
    context: str | None = None
    problem_spec: str | None = None
    generated_spec: str | None = None
    equivalence: str | None = None

    @classmethod
    def from_file(cls, filepath: Path) -> "CoqFile":
        content = filepath.read_text()

        def extract_section(section_name: str) -> str | None:
            pattern = (
                rf"\(\* start {section_name} \*\)(.*?)\(\* end {section_name} \*\)"
            )
            match = re.search(pattern, content, re.DOTALL)
            return match.group(1).strip() if match else None

        return cls(
            filepath=filepath,
            full_content=content,
            prompt=extract_section("prompt"),
            imports=extract_section("imports"),
            context=extract_section("context"),
            problem_spec=extract_section("problem_spec"),
            generated_spec=extract_section("generated_spec"),
            equivalence=extract_section("equivalence"),
        )

    def nl_description_prompt(self) -> tuple[str, str]:
        if not self.prompt:
            raise ValueError("empty prompt")

        prompt_content = self.prompt.strip()

        lines = []
        for line in prompt_content.split("\n"):
            line = line.strip()
            if line.startswith("(*") or line.startswith("*)"):
                continue
            if line.startswith("*"):
                line = line[1:].strip()
            lines.append(line)

        prompt_text = "\n".join(lines).strip()

        signature = ""
        description = ""
        examples = []
        current_section = None

        for line in prompt_text.split("\n"):
            line = line.strip()
            if line.startswith("function_signature:"):
                signature = line.replace("function_signature:", "").strip().strip('"')
                current_section = "signature"
            elif line.startswith("docstring:"):
                description = line.replace("docstring:", "").strip()
                current_section = "docstring"
            elif line.startswith("examples:"):
                current_section = "examples"
            elif line.startswith("- input:"):
                input_part = line.replace("- input:", "").strip()
                examples.append(f"- Input: {input_part}")
            elif line.startswith("expected_output:"):
                output_part = line.replace("expected_output:", "").strip()
                if examples:
                    examples[-1] += f" -> Output: {output_part}"
            elif (
                current_section == "function_signature"
                and line
                and not line.startswith(("-", "docstring:", "examples:"))
            ):
                signature += " " + line
            elif (
                current_section == "docstring"
                and line
                and not line.startswith(("-", "function_signature:", "examples:"))
            ):
                description += " " + line

        description = description.strip().strip('|').strip()
        formatted_prompt = f"Signature:\n{signature}\n\nDescription:\n{description}"

        if examples:
            formatted_prompt += "\n\nExamples:"
            for example in examples:
                formatted_prompt += f"\n{example}"

        context_part = ""
        if self.context:
            context_part = f"\n\nGiven types, definitions, and parameters:\n{self.context}"

        imports_part = ""
        if self.imports:
            imports_part = f"\n\nImports available:\n{self.imports}"

        return f"{formatted_prompt}{imports_part}{context_part}", f"{imports_part}{context_part}"
