# -*- coding: utf-8 -*-
"""
Created on Mon Aug  5 11:29:14 2024

@author: kernel
"""
import time
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
from sklearn.preprocessing import MinMaxScaler
from sklearn.metrics import mean_squared_error, mean_absolute_error
from tensorflow.keras.models import Sequential
from tensorflow.keras.layers import Dense, Layer
from tensorflow.keras.initializers import glorot_uniform, orthogonal, zeros
from tensorflow.keras import backend as K
import tensorflow as tf
plt.rcParams['figure.dpi'] = 600
# 自定义复杂线性层


class CustomLSTMCell(Layer):
    def __init__(self, units, **kwargs):
        self.units = units
        self.state_size = [units, units]
        super(CustomLSTMCell, self).__init__(**kwargs)

    def build(self, input_shape):
        self.kernel = self.add_weight(shape=(input_shape[-1], self.units * 4),
                                      initializer=glorot_uniform(),
                                      name='kernel')
        self.recurrent_kernel = self.add_weight(shape=(self.units, self.units * 4),
                                                initializer=orthogonal(),
                                                name='recurrent_kernel')
        self.bias = self.add_weight(shape=(self.units * 4,),
                                    initializer=zeros(),
                                    name='bias')
        self.built = True

    def call(self, inputs, states):
        h_tm1, c_tm1 = states  # previous memory state and carry state

        z = K.dot(inputs, self.kernel) + K.dot(h_tm1, self.recurrent_kernel) + self.bias

        z0, z1, z2, z3 = tf.split(z, num_or_size_splits=4, axis=1)

        i = K.sigmoid(z0)
        f = K.sigmoid(z1)
        c = f * c_tm1 + i * K.tanh(z2)
        o = K.sigmoid(z3)
        h = o * K.tanh(c)

        return h, [h, c]

# 自定义LSTM层
class CustomLSTM(Layer):
    def __init__(self, units, return_sequences=True, return_state=False, **kwargs):
        self.units = units
        self.return_sequences = return_sequences
        self.return_state = return_state
        super(CustomLSTM, self).__init__(**kwargs)
        self.cell = CustomLSTMCell(units, input_dim=1)

    def build(self, input_shape):
        self.cell.build(input_shape)
        self.built = True

    def call(self, inputs, initial_state=None):
        if initial_state is None:
            initial_state = self.get_initial_state(inputs)
        
        states = initial_state
        outputs = []

        for t in range(inputs.shape[1]):
            output, states = self.cell(inputs[:, t, :], states)
            if self.return_sequences:
                outputs.append(output)

        if self.return_sequences:
            outputs = tf.stack(outputs, axis=1)
        else:
            outputs = output

        if self.return_state:
            return [outputs] + states
        else:
            return outputs

    def get_initial_state(self, inputs):
        batch_size = tf.shape(inputs)[0]
        initial_state = [tf.zeros((batch_size, self.units)) for _ in range(2)]
        return initial_state

# 下载和加载数据集
url = 'https://raw.githubusercontent.com/jbrownlee/Datasets/master/airline-passengers.csv'
df = pd.read_csv(url, usecols=[1])
# df = pd.read_csv("D:\大湾区group\时间序列\XNet在低维函数逼近上的应用\data_generator\model_1.csv")
data = df.values.astype('float32')

# 数据归一化
scaler = MinMaxScaler(feature_range=(0, 1))
scaled_data = scaler.fit_transform(data)

# 创建训练和测试数据
# train_size = int(len(scaled_data) * 0.7)
train_size = int(len(scaled_data) - 36)
train, test = scaled_data[0: train_size + 24], scaled_data[train_size:len(scaled_data)]

def create_dataset(dataset, look_back=24, forecast_horizon=12):
    X, Y = [], []
    for i in range(len(dataset) - look_back - forecast_horizon + 1):
        X.append(dataset[i:i + look_back, 0])
        Y.append(dataset[i + look_back:i + look_back + forecast_horizon, 0])
    return np.array(X), np.array(Y)

look_back = 24
forecast_horizon=12
X_train, y_train = create_dataset(train, look_back, forecast_horizon)
X_test, y_test = create_dataset(test, look_back, forecast_horizon)

# 将数据重塑为 [samples, time steps, features] 格式
X_train = np.reshape(X_train, (X_train.shape[0], X_train.shape[1], 1))
X_test = np.reshape(X_test, (X_test.shape[0], X_test.shape[1], 1))

# 创建和训练LSTM模型
model = Sequential()
model.add(CustomLSTM(64, input_shape=(look_back, 1), return_sequences=False))
model.add(Dense(forecast_horizon))
model.compile(optimizer='adam', loss='mean_squared_error')

# 定义回调函数，每 100 个 epoch 输出一次日志
class CustomCallback(tf.keras.callbacks.Callback):
    def __init__(self):
        super().__init__()
        self.losses = []  # 初始化用于存储loss值的列表

    def on_epoch_end(self, epoch, logs=None):
        # 保存loss值
        self.losses.append(logs['loss'])
        # 每100 epochs打印一次信息
        if (epoch + 1) % 100 == 0:
            print(f"Epoch {epoch + 1}, Loss: {logs['loss']:.4e}")  # 使用科学计数法显示loss

# 训练模型
callback = CustomCallback()
start_time = time.time()
model.fit(X_train, y_train, epochs=5000, batch_size=32, verbose=0, callbacks=[callback])
elapsed_time = time.time() - start_time

# 进行预测
train_predict = model.predict(X_train)
test_predict = model.predict(X_test)

# 逆归一化
train_predict = scaler.inverse_transform(train_predict)
test_predict = scaler.inverse_transform(test_predict)
y_train = scaler.inverse_transform(y_train)
y_test = scaler.inverse_transform(y_test)

# 计算误差
mse = mean_squared_error(y_test, test_predict)
rmse = np.sqrt(mse)
mae = mean_absolute_error(y_test, test_predict)
mae_train = mean_absolute_error(y_train, train_predict)

print(f"Mean Squared Error (MSE) of validation data: {mse:.4e}")
print(f"Root Mean Squared Error (RMSE) of validation data: {rmse:.4e}")
print(f"LSTM Mean Absolute Error (MAE) of validation data: {mae:.4e}")
print(f"LSTM Mean Absolute Error (MAE) of training data: {mae_train:.4e}")
print(f"Training time: {elapsed_time:.4f}")

train_predict_last = train_predict[:, -1]  # 形状为 (65,)
test_predict_last = test_predict    # 形状为 (28,)

# 绘制结果
# train_plot = np.empty_like(data)
# train_plot[:] = np.nan
# train_plot[look_back:len(train_predict_last) + look_back, 0] = train_predict_last.flatten()

test_plot = np.empty_like(data)
test_plot[:] = np.nan
test_plot[- forecast_horizon :len(data), 0] = test_predict_last.flatten()

# 绘图
plt.figure(figsize=(10, 6))
plt.plot(data, label='Original Data', color='dodgerblue', linewidth=2)
# plt.plot(train_plot, label='Train Prediction (Last Step)', color='coral', linestyle='--', linewidth=2)
plt.plot(test_plot, label='Test Prediction (Last Step)', color='limegreen', linestyle='--', linewidth=2)
plt.title('Model Predictions vs Original Data', fontsize=16)
plt.ylabel('LSTM', fontsize=14)
plt.xlabel('Time Steps', fontsize=14)
plt.legend(loc='upper left', fontsize=12)
plt.grid(True, linestyle='--', alpha=0.7)
plt.savefig(r'D:\大湾区group\lixin\neuralforecast-main\loss\LSTM_Model1.png')
plt.show()



plt.figure(figsize=(10, 6))
plt.plot(callback.losses, label='Training Loss by LSTM', alpha=0.5)
# plt.title('Evolution of Training Loss Over Epochs', fontsize=16)
plt.xlabel('Epochs', fontsize=14)
plt.ylabel('Training Loss', fontsize=14)
plt.yscale('log')
plt.yticks([1, 0.1, 0.01, 0.001, 0.0001], ['$10^0$', '$10^{-1}$', '$10^{-2}$', '$10^{-3}$', '$10^{-4}$'])
plt.legend()
plt.grid(True, linestyle='--', alpha=0.7)
plt.savefig(r'D:\大湾区group\lixin\neuralforecast-main\loss\LSTM_Model1_loss.png')  
plt.show()
np.save('D:/大湾区group/lixin/neuralforecast-main/loss/LSTM_loss_model1.npy', np.array(callback.losses))




