import numpy as np

from scipy.special import bernoulli, binom


def spline_kernel(x, y, q):
    assert q % 2 == 0, f"q needs to be even: {q}"
    assert q > 1
    polynom = bernoulli_polynomial(q)
    return 1 + (-1)**(q//2 - 1)/(np.math.factorial(q)) * polynom(np.abs(x - y))


def bernoulli_polynomial_high(n: int):
    def f(x: np.ndarray) -> np.ndarray:
        bernoulli_coeff = bernoulli(n)
        result = np.zeros_like(x)
        # No broadcasting to avoid memory issues 
        # einsum could be used, but the python loop is negligible compared to the matrix power.
        for k in range(n+1):
            result += bernoulli_coeff[n-k] * binom(n, k) * x ** k
        return result
    return f


def bernoulli_polynomial(n: int):
    if n == 0:
        def f(x):
            return np.ones_like(x)
    elif n == 1:
        def f(x):
            return x - 1/2
    elif n == 2:
        def f(x):
            return x**2 - x + 1/6
    elif n == 3:
        def f(x):
            return x**3 - 3/2 * x**2 + 1/2 * x
    elif n == 4:
        def f(x):
            return x**4 - 2 * x**3 + x ** 2 - 1/30
    elif n == 5:
        def f(x):
            return x**5 - 5/2 * x**4 + 5/3 * x**3 - 1/6 * x
    elif n == 6:
        def f(x):
            return x ** 6 - 3 * x ** 5 + 5/2 * x**4 - 1/2 * x**2 + 1/42
    else:
        return bernoulli_polynomial_high(n)
    return f
