import abc
from typing import Tuple

import numpy as np


class Loss(metaclass=abc.ABCMeta):
    @abc.abstractmethod
    def f(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        """Function value."""
        pass

    @abc.abstractmethod
    def df(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        """Function gradient."""
        pass

    @abc.abstractmethod
    def ddf(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        """Function value, gradient and Hessian in a row."""
        pass

    @abc.abstractmethod
    def all_derivatives(self, y: np.ndarray, y_pred: np.ndarray) -> Tuple[np.ndarray]:
        """Function value, gradient and Hessian in a row."""
        pass


class LogisticLoss(Loss):
    """Logistic Loss function. Values and 1,2 derivatives."""

    def w(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        return np.exp(- y * y_pred)

    def f(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        return np.log(1 + self.w(y, y_pred))

    def df(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        w = self.w(y, y_pred)
        return - y * w / (1 + w)

    def ddf(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        w = self.w(y, y_pred)
        return y**2 * w / (1 + w)**2

    def all_derivatives(self, y: np.ndarray, y_pred: np.ndarray) -> Tuple[np.ndarray]:
        w = self.w(y, y_pred)
        f = np.log(1 + w)
        df = - y * w / (1 + w)
        ddf = y**2 * w / (1 + w)**2
        return f, df, ddf


class HuberLoss(Loss):
    """Huber Loss function. Values and 1,2 derivatives."""

    def w(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        return 1 + (y_pred - y)**2

    def f(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        return np.sqrt(self.w(y, y_pred)) - 1

    def df(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        w = self.w(y, y_pred)
        return (y_pred - y) / np.sqrt(w)

    def ddf(self, y: np.ndarray, y_pred: np.ndarray) -> np.ndarray:
        w = self.w(y, y_pred)
        return w**(-3/2)

    def all_derivatives(self, y: np.ndarray, y_pred: np.ndarray) -> Tuple[np.ndarray]:
        w = self.w(y, y_pred)
        f = np.sqrt(w) - 1
        df = (y_pred - y) / np.sqrt(w)
        ddf = w**(-3/2)
        return f, df, ddf
