# FeatHawkes  
*Fast, feature-aware multivariate Hawkes processes in PyTorch*  

<p align="center">
  <img src="https://raw.githubusercontent.com/CONFIDENTIAL/hawkes-exp/main/docs/_static/logo.svg" alt="FeatHawkes logo" width="220"/>
</p>

---

## ✨ Why FeatHawkes?

* **Simulation that scales.** Vectorised Ogata thinning lets you generate thousands of events in seconds.  
* **Inference you can trust.** Maximum-likelihood estimation is implemented in pure PyTorch — autograd-friendly, GPU-accelerated, fully differentiable.  
* **Feature-dependent dynamics.** Excitation matrix α can be learned as a logistic function of *node-wise* exogenous features (one or *many*).  
* **Zero-friction API.** Fit from a **single flattened list** of events *or* from **one list per trajectory** – we pad and batch for you.  
* **Batteries included.** Clean visualisation helpers, reproducible notebooks, and exhaustive tests.
* **REMARQ.** For now, the team features are integrated with the player features for simplicity of use in the application (the same feature is applied to each player in a team). Currently, the associated theta1 and theta2 coefficients for these features need to be added together. A future version will include a specific team vector


## 📋 Table of Contents

1. [Installation](#installation)  
2. [Quick start](#quick-start)  
3. [Two ways to feed your data](#two-ways-to-feed-your-data)  
4. [Fit with or without features](#fit-with-or-without-features)  
5. [API reference](#api-reference)   
6. [Project structure](#project-structure)  
7. [Citing FeatHawkes](#citing-feathawkes)  

---

## 📦 Installation

```bash
git clone https://github.com/CONFIDENTIAL/FeatHawkes.git
cd FeatHawkes


**Requirements:** FeatHawkes supports Python 3.9+ and PyTorch ≥ 2.2. CUDA is detected automatically; if a GPU is available, everything (simulation and training) will run on it by default.

## 🚀 Quick start

```python
import numpy as np
from openhawkes.simulation import ExpHawkes
from openhawkes.visualization import plot_intensity_and_jumps_same_yscale
from openhawkes.inference import fit_hawkes

np.set_printoptions(precision=2, suppress=True)

# ----- 1. define a 3-dim exponential Hawkes process
mu    = [0.10, 0.15, 0.20]                       # base intensities
alpha = [[0.50, 0.00, 0.00],                    # excitation matrix
         [0.00, 0.00, 0.50],
         [0.10, 0.20, 0.00]]
beta  = [1.0, 2.0, 1.0]                         # decays (1 per dim)

proc = ExpHawkes(mu, alpha, beta, seed=0)

# ----- 2. simulate 1 000 s of activity
events = proc.simulate(max_time=1_000)
print(f"Generated {len(events.times)} events.")

# ----- 3. visualise intensities + jumps
plot_intensity_and_jumps_same_yscale(proc)

# ----- 4. infer parameters back (vectorised MLE, no features)
model = fit_hawkes(
    nb_types=len(mu),
    times=events.times,          # flat list of timestamps
    types=events.types,          # flat list of types (0…D-1)
    nb_batches=10,               # split the long list into 10 mini-batches
    beta_unique=False            # learn one β per dimension
)

print("\n--- Recovered parameters ---")
print("μ =", model.mu.detach().cpu().numpy())
print("α =", model.alpha.detach().cpu().numpy())
print("β =", model.beta.detach().cpu().numpy())
```

## 🥢 Two ways to feed your data

`fit_hawkes` now accepts either of the following:

| Interface | What you pass | When to use |
|-----------|---------------|-------------|
| **Flat lists (legacy)** | `times`, `types` (both flat) + `nb_batches` | You already have a single concatenated list of events. |
| **List of lists** | `seq_times`, `seq_types` (one list per trajectory) | Natural when events are grouped by match / user / session. Automatic padding & batching. |

Optional per-trajectory features live in `seq_feats` (shape `(S, D_feat)` or `(S, K, D_feat)` for K features).

```python
# multiple independent sequences + features
traj_times  = [[0.2, 0.5, 1.1], [0.1, 0.8]]      # len = S = 2
traj_types  = [[0,   1,   2  ], [1,   1 ]]
traj_feats  = np.array([[1.2, 0.3, -0.7],        # shape (S, D_feat)
                        [0.8, 0.1,  0.4]])

mdl = fit_hawkes(
    nb_types=3,
    seq_times=traj_times,
    seq_types=traj_types,
    seq_feats=traj_feats,        # features **per sequence**
    nb_of_features=1,            # because feats.shape = (S, D_feat)
)
```

The legacy `(times, types)` path is fully backward-compatible.

## 🔎 Fit with or without features

### 1. Constant-α Hawkes (no features)
```python
model = fit_hawkes(
    nb_types=4,
    times=events.times,
    types=events.types,
    beta_unique=True           # single β for all dims
)
```

### 2. Logistic α = σ(γ + θ₁·xᵢ + θ₂·xⱼ) for one feature
```python
model = fit_hawkes(
    nb_types=4,
    seq_times=seq_times,
    seq_types=seq_types,
    seq_feats=seq_feat_vecs,   # shape (S, D)  – one feature / dim
    nb_of_features=1,          # <-- 1 !
)
```

### 3. Multiple features (matrix of K features per dim)
```python
model = fit_hawkes(
    nb_types=4,
    seq_times=seq_times,
    seq_types=seq_types,
    seq_feats=seq_feat_tensor,   # shape (S, K, D)
    nb_of_features=K,            # <-- K !
    lambda_l1=5e-3,              # optional L¹ sparsity on θ
)
```

Under the hood we use `HawkesMLE`, `HawkesMultiFeatureMLE` depending on whether `seq_feats` is None, 2-D or 3-D.

> **Note:** The EM and slow-loop Ogata baselines remain in the code for reference (`model_type="em"` or `"slow_loop_ogata"`), but they are not documented here to keep the README focused on the recommended vectorised path.

## 🧮 API reference

### `fit_hawkes` (high-level helper)

| Argument | Type | Default | Description |
|----------|------|---------|-------------|
| **Data** | | | |
| `seq_times` | `Sequence[Sequence[float]]` | — | List of timestamp lists (one per trajectory). |
| `seq_types` | `Sequence[Sequence[int]]` | — | Parallel list of type indices. |
| `seq_feats` | `np.ndarray` | `None` | Sequence-level features. Shape `(S, D_feat)` for 1 feature or `(S, K, D_feat)` for K features. |
| `times` | `Sequence[float]` | — | Flat list of timestamps (legacy). |
| `types` | `Sequence[int]` | — | Flat list of types (legacy). |
| `nb_batches` | `int` | `1` | How many mini-batches to split the flat list into. |
| **Optim / hyper-parameters** | | | |
| `max_epochs` | `int` | `10_000` | SGD iterations (Adam). |
| `lr` | `float` | `5e-3` | Initial learning rate. |
| `beta_unique` | `bool` | `True` | Fit a scalar β if True; else one β per dimension. |
| `nb_of_features` | `int` | `1` | Only relevant when `seq_feats` is given. |
| `lambda_l1` | `float` | `0.0` | L¹ penalty on α or θ (promotes sparsity). |
| **Training tricks** | | | |
| `epsilon` / `start_plateau` / `epsilon_plateau` / `adjust_lr` | — | — | Controls the built-in plateau LR scheduler. |
| `log_b`, `factor_b` | — | — | Adds a positivity log-barrier on μ and β. |
| **System** | | | |
| `device` | `torch.device` | auto | If None, CUDA if available else CPU. |

Returned object always has `.mu`, `.alpha` (or `.gamma`,`.theta1`,`.theta2`) `.beta` as learned `torch.nn.Parameters`, plus any additional θ if features were used.

### `ExpHawkes`
Fast simulator based on Ogata thinning. Accepts vectors/matrix μ, α, β.

### `plot_intensity_and_jumps_same_yscale`
One-liner that draws (i) the true intensity λₖ(t) of each dimension and (ii) tick marks for jump times, all on a shared y-axis for easy comparison.


## 🗂 Project structure

```
FeatHawkes/
├── src/openhawkes/
│   ├── simulation.py        
│   ├── inference.py         
│   ├── models.py             
│   ├── visualization.py      
│   ├── utils.py             
│   └── __init__.py
├── examples/
│   ├── quickstart.py
│   ├── simulate.py
│   └── quickstart_with_features.py
├── notebooks/
│   ├── paper_ICLR
│   └── just_try/
│       ├── with_features.ipynb
│       └── without_features.ipynb
└── tests/
    └── test_simulation.py
```

## 🙏 Citing FeatHawkes

```bibtex
@software{feat_hawkes_2025,
  author  = {Anonymous},
  title   = {FeatHawkes: Feature-Aware Hawkes Processes in PyTorch},
  year    = 2025,
  url     = {https://github.com/CONFIDENTIAL/hawkes-exp},
  note    = {Version 0.1.0}
}
```
