#!/usr/bin/env python3
""" 
Demo script: generate synthetic Hawkes data with exogenous features,
then fit a feature‑dependent Hawkes model.
"""
import numpy as np
from openhawkes.utils       import generate_alphas_with_features
from openhawkes.simulation  import ExpHawkes
from openhawkes.inference   import fit_hawkes

# ── 1) True parameters ───────────────────────────────────────────────
D           = 3
nb_sequences = 5
max_events    = 500
mu_true    = np.array([0.1,  0.15, 0.2])
beta_true  = np.array([0.8, 1, 1.2])    # β unique

gamma = np.array([
    [-1.0, -0.5, -1.0],
    [-1.0, -0.5, -1.0],
    [-0.5,  0.0,  0.0]
])
theta1 = np.array([
    [ 0.0,   0.0,   0.0],
    [ 0.0,  -0.75,  0.50],
    [ 0.0,  -1.00,  0.0]
])
theta2 = np.array([
    [ 0.0,   0.0,   0.75],
    [ 0.50,  0.0,  -0.50],
    [ 0.75, -0.25,  0.0]
])

# ── 2) Generate data ─────────────────────────────────────────────────
rng = np.random.default_rng(42)
all_times, all_types, features = [], [], []

for _ in range(nb_sequences):
    alpha_mat, x_vec = generate_alphas_with_features(rng, gamma, theta1, theta2)
    features.append(x_vec)

    proc = ExpHawkes(mu_true, alpha_mat, beta_true, seed=None)
    ev   = proc.simulate(max_events=max_events)
    all_times.extend(ev.times)
    all_types.extend(ev.types)

print(f"Generated a total of {len(all_times)} events from {nb_sequences} sequences.")

# ── 3) Fit feature‑dependent Hawkes ───────────────────────────────────
modelF = fit_hawkes(
    nb_types = D,
    times = all_times,
    types = all_types,
    seq_feats = features,
    lr = 0.01,
    nb_batches = nb_sequences,
    beta_unique = False
)

# ── 4) Print recovered parameters ────────────────────────────────────
print("\n--- Recovered parameters ---")
print("μ  =", modelF.mu.detach().cpu().numpy())
print("β  =", modelF.beta.detach().cpu().numpy())
print("B0 =", modelF.gamma.detach().cpu().numpy())
print("B1 =", modelF.theta1.detach().cpu().numpy())
print("B2 =", modelF.theta2.detach().cpu().numpy())
