#!/usr/bin/env python3 
"""
CLI script to simulate a Hawkes process and save the events to JSON.

Examples:
    # Simulate a 3‐dimensional process until t=10 and write to data.json
    python examples/simulate.py --d 3 --max-time 10 data.json

    # Simulate until 500 jumps, default d=2
    python examples/simulate.py --max-jumps 500 events.json
"""
import json
import click
import numpy as np
from openhawkes.simulation import ExpHawkes

@click.command()
@click.argument("outfile", type=click.Path(writable=True))
@click.option("--d", "-d", type=int, default=2,
              help="Dimension of the Hawkes process")
@click.option("--max-time", "-T", type=float, default=None,
              help="Simulate until this time horizon")
@click.option("--max-events", "-N", type=int, default=None,
              help="Simulate until this many jumps")
@click.option("--seed", type=int, default=None,
              help="Random seed for reproducibility")
def main(outfile, d, max_time, max_events, seed):
    # Default parameters (adaptés selon la dimension d)
    mu = [0.1, 0.15, 0.2]
    
    # Créer une matrice alpha de dimension d×d
    alpha = np.zeros((d, d))
    if d >= 2:
        alpha[0, 0] = 0.5
    if d >= 3:
        alpha[1, 2] = 0.5
        alpha[2, 0] = 0.1
        alpha[2, 1] = 0.2
    
    beta = [1.0]
    
    # Instantiate & simulate
    proc = ExpHawkes(mu, alpha, beta, seed=seed)
    events = proc.simulate(max_time=max_time, max_events=max_events)
    
    # Write JSON
    with open(outfile, "w") as f:
        json.dump({"times": events.times, "types": events.types}, f)
    
    print(f"✔ Wrote {len(events.times)} events to {outfile}")

if __name__ == "__main__":
    main()