import os
import yaml
import time
import random
import pickle
import numpy as np

import torch
from torch.utils.data import DataLoader

from glob import glob
from tqdm import tqdm
from easydict import EasyDict

from utils.utils import AdvectionDataset
from models.diffusion import Diffusion

__all__ = [
    'sampling_eval'
]

def get_new_work_dir(root: str, config_name: str) -> str:
    fn = time.strftime('%Y_%m_%d__%H_%M_%S', time.localtime())
    if config_name.startswith('/'):
        log_dir = root + config_name + '-' + fn
    else:
        log_dir = root + '-'  + config_name + '-' + fn
    os.makedirs(log_dir)
    return log_dir
        
def seed_all(seed: int = 42) -> None:
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.backends.cudnn.deterministic = True
    torch.use_deterministic_algorithms = True
        
def sampling_eval(ckpt_path: str, method: str = None) -> str:
    ckpt = torch.load(ckpt_path)
    config_path = glob(os.path.join(os.path.dirname(os.path.dirname(ckpt_path)), '*.yml'))[0]
    with open(config_path, 'r') as f:
        config = EasyDict(yaml.safe_load(f))
    seed_all(config.train.seed)
    device = torch.device(config.model.device)

    if method is None:
        method = config.model.sampling.method

    output_dir = get_new_work_dir(os.path.dirname(os.path.dirname(ckpt_path)), f'/{method}_predict_on_xt')

    test_ratio = 0.1
    initial_step, reduced_batch = 1, 1
    reduced_resolution_t, reduced_resolution = 5, 4
    test_data = AdvectionDataset(
        config.dataset_path,
        initial_step=initial_step,
        reduced_resolution=reduced_resolution,
        reduced_resolution_t=reduced_resolution_t,
        reduced_batch=reduced_batch,
        test_ratio=test_ratio,
        if_test=True,
    )

    test_loader = DataLoader(
        test_data,
        config.train.batch_size,
        num_workers=0,
        shuffle=False
    )

    model = Diffusion(config.model, dx=test_data.dx, dt=test_data.dt).to(device)
    model.load_state_dict(ckpt['model'])

    
    gt_func_values, results = [], []
    pbar = tqdm(test_loader, dynamic_ncols=True)
    for i, (condition, feature, grid) in enumerate(pbar):
        gt_func_values.append(feature)
        results.append(
            model.sampling(
                feature.shape,
                condition.to(device),
                grid.to(device),
                method
            ).detach().cpu().numpy()
        )

        with open(os.path.join(output_dir, 'samples_%d.pkl' % i), 'wb') as f:
            pickle.dump(np.concatenate(results), f)

        test_loss = np.round(
            np.sqrt(np.square(np.concatenate(results) - np.concatenate(gt_func_values)).mean()),
            decimals=5
        )
        pbar.set_description(f'Test loss: {test_loss}')

    for temp_file in os.listdir(output_dir):
        os.remove(os.path.join(output_dir, temp_file))

    samples_path = os.path.join(output_dir, f'samples_all_{test_loss}.pkl')
    with open(samples_path, 'wb') as f:
        pickle.dump(np.concatenate(results), f)

    return samples_path
        
if __name__ == '__main__':
    torch.multiprocessing.set_start_method('spawn') 

    model_path = 'logs/GRU/pde_1.0_0.1---GRU---hidden_size-512--n_layers-3---2024_07_04__00_51_46/checkpoints/ckpt.pt'

    method = 'ode'
    sampling_eval(model_path)
