"""
Script to convert R1 hdf5 data to the LeRobot dataset v2.0 format.

Example usage: uv run examples/r1/convert_r1_data_to_lerobot_single.py --raw-dir /path/to/raw/data --repo-id <org>/<dataset-name>
"""

import dataclasses
from pathlib import Path
import shutil
from typing import Literal
import random
import h5py
from lerobot.common.datasets.lerobot_dataset import HF_LEROBOT_HOME
from lerobot.common.datasets.lerobot_dataset import LeRobotDataset
import numpy as np
import torch
import tqdm
import tyro


@dataclasses.dataclass(frozen=True)
class DatasetConfig:
    use_videos: bool = True
    tolerance_s: float = 0.0001
    image_writer_processes: int = 10
    image_writer_threads: int = 5
    video_backend: str | None = None


DEFAULT_DATASET_CONFIG = DatasetConfig()


def create_empty_dataset(
    repo_id: str,
    robot_type: str,
    mode: Literal["video", "image"] = "video",
    *,
    has_velocity: bool = False,
    has_effort: bool = False,
    dataset_config: DatasetConfig = DEFAULT_DATASET_CONFIG,
) -> LeRobotDataset:
    motors = [
        "left_arm_0",
        "left_arm_1",
        "left_arm_2",
        "left_arm_3",
        "left_arm_4",
        "left_arm_5",
        "left_gripper",
        "mobile_base_0",
        "mobile_base_1",
        "mobile_base_2",
        "right_arm_0",
        "right_arm_1",
        "right_arm_2",
        "right_arm_3",
        "right_arm_4",
        "right_arm_5",
        "right_gripper",
        "torso_0",
        "torso_1",
        "torso_2",
        "torso_3",
    ]
    action_motors = [
        'left_arm_0',
        'left_arm_1',
        'left_arm_2',
        'left_arm_3',
        'left_arm_4',
        'left_arm_5',
        'left_gripper',
        'mobile_base_0',
        'mobile_base_1',
        'mobile_base_2',
        'right_arm_0',
        'right_arm_1',
        'right_arm_2',
        'right_arm_3',
        'right_arm_4',
        'right_arm_5',
        'right_gripper',
        'torso_0',
        'torso_1',
        'torso_2',
        'torso_3',
    ]
    cameras = [
        "head",
        "left_wrist",
        "right_wrist",
    ]

    features = {
        "observation.state": {
            "dtype": "float32",
            "shape": (len(motors),),
            "names": [
                motors,
            ],
        },
        "action": {
            "dtype": "float32",
            "shape": (len(action_motors),),
            "names": [
                action_motors
            ],
        },
    }

    # Add camera features with simplified structure
    for cam in cameras:
        features[f"observation.images.{cam}"] = {
            "dtype": mode,
            "shape": (270, 480, 3),
            "names": [
                "channels",
                "height",
                "width",
            ],
        }

    if Path(HF_LEROBOT_HOME / repo_id).exists():
        shutil.rmtree(HF_LEROBOT_HOME / repo_id)

    return LeRobotDataset.create(
        repo_id=repo_id,
        fps=50,
        robot_type=robot_type,
        features=features,
        use_videos=dataset_config.use_videos,
        tolerance_s=dataset_config.tolerance_s,
        image_writer_processes=dataset_config.image_writer_processes,
        image_writer_threads=dataset_config.image_writer_threads,
        video_backend=dataset_config.video_backend,
    )


def load_raw_images_per_camera(ep: h5py.File, cameras: list[str]) -> dict[str, np.ndarray]:
    imgs_per_cam = {}

    for camera in cameras:
        # print(ep, camera),quit()
        uncompressed = ep[f"/obs/rgb/{camera}/img"].ndim == 4

        if uncompressed:
            # load all images in RAM
            imgs_array = ep[f"/obs/rgb/{camera}/img"][:]
        else:
            import cv2

            # load one compressed image after the other in RAM and uncompress
            imgs_array = []
            for data in ep[f"/obs/rgb/{camera}/img"]:
                img = cv2.imdecode(data, 1)
                imgs_array.append(img)
            imgs_array = np.array(imgs_array)

        imgs_per_cam[camera] = imgs_array
    return imgs_per_cam

def load_state(ep: h5py.File):
    states = []
    states.append(ep['obs/joint_state/left_arm/joint_position'][:, :6])
    states.append(ep['obs/gripper_state/left_gripper/gripper_position'][:].reshape(-1, 1))
    states.append(ep['obs/chassis_odom/linear_velocity'][:, :2])
    states.append(ep['obs/chassis_odom/angular_velocity'][:, -1:])
    states.append(ep['obs/joint_state/right_arm/joint_position'][:, :6])
    states.append(ep['obs/gripper_state/right_gripper/gripper_position'][:].reshape(-1, 1))
    states.append(ep['obs/joint_state/torso/joint_position'][:])

    states = np.concatenate(states, axis=-1)
    return torch.from_numpy(states)

def load_action(ep: h5py.File): 
    action_keys = [
        'action/left_arm', 
        'action/left_gripper', 
        'action/mobile_base', 
        'action/right_arm', 
        'action/right_gripper', 
        'action/torso'
    ]   
    actions = [ep[k][:] for k in action_keys]
    actions = np.concatenate(actions, axis=-1)
    return torch.from_numpy(actions)

def load_raw_episode_data(
    ep_path: Path,
) -> tuple[dict[str, np.ndarray], torch.Tensor, torch.Tensor, torch.Tensor | None, torch.Tensor | None]:
    with h5py.File(ep_path, "r") as ep:
        # state = torch.from_numpy(ep["/observations/qpos"][:])
        # action = torch.from_numpy(ep["/action"][:])
        state = load_state(ep)
        action = load_action(ep)

        velocity = None
        # if "/observations/qvel" in ep:
        #     velocity = torch.from_numpy(ep["/observations/qvel"][:])

        effort = None
        # if "/observations/effort" in ep:
        #     effort = torch.from_numpy(ep["/observations/effort"][:])

        imgs_per_cam = load_raw_images_per_camera(
            ep,
            [
                "head",
                "left_wrist",
                "right_wrist",
            ],
        )

    return imgs_per_cam, state, action, velocity, effort


def populate_dataset(
    dataset: LeRobotDataset,
    hdf5_files: list[Path],
    task: str,
    episodes: list[int] | None = None,
) -> LeRobotDataset:
    if episodes is None:
        episodes = range(len(hdf5_files))

    for ep_idx in tqdm.tqdm(episodes):
        ep_path = hdf5_files[ep_idx]

        imgs_per_cam, state, action, velocity, effort = load_raw_episode_data(ep_path)
        num_frames = state.shape[0]

        for i in range(num_frames):
            frame = {
                "observation.state": np.float32(state[i]),
                "action": np.float32(action[i]),
                "task": task,
            }

            for camera, img_array in imgs_per_cam.items():
                frame[f"observation.images.{camera}"] = img_array[i]

            dataset.add_frame(frame)

        dataset.save_episode()

    return dataset


def port_r1(
    raw_dir: Path,
    repo_id: str,
    raw_repo_id: str | None = None,
    task: str = "DEBUG",
    num_episodes: int = -1,
    *,
    episodes: list[int] | None = None,
    push_to_hub: bool = True,
    success_only: bool = False,
    mode: Literal["video", "image"] = "image",
    dataset_config: DatasetConfig = DEFAULT_DATASET_CONFIG,
):
    if (HF_LEROBOT_HOME / repo_id).exists():
        shutil.rmtree(HF_LEROBOT_HOME / repo_id)

    postfix = 'success' if success_only else ''
    raw_hdf5_files = sorted(raw_dir.glob(f"collected_data_*{postfix}.h5"))
    
    if num_episodes > 0:
        # use part of the data
        assert len(raw_hdf5_files) >= num_episodes
        hdf5_files = np.random.choice(raw_hdf5_files, num_episodes, replace=False)
    else:
        # use all data
        hdf5_files = raw_hdf5_files

    dataset = create_empty_dataset(
        repo_id,
        robot_type="r1",
        mode=mode,
        has_effort=False,
        has_velocity=False,
        dataset_config=dataset_config,
    )
    dataset = populate_dataset(
        dataset,
        hdf5_files,
        task=task,
        episodes=episodes,
    )

    if push_to_hub:
        dataset.push_to_hub()


if __name__ == "__main__":
    tyro.cli(port_r1)
