// import { JsPsych, JsPsychPlugin, ParameterType, TrialType } from "jspsych";

let jsPsychHtmlMultiSliderResponse = (function (jspsych) {
    let ParameterType = jspsych.ParameterType;
    const info = {
        name: "html-slider-response",
        parameters: {
            /** The HTML string to be displayed */
            stimulus: {
                type: ParameterType.HTML_STRING,
                pretty_name: "Stimulus",
                default: undefined,
            },
            /** Sets the minimum value of the slider. */
            min: {
                type: ParameterType.INT,
                pretty_name: "Min slider",
                default: 0,
            },
            /** Sets the maximum value of the slider */
            max: {
                type: ParameterType.INT,
                pretty_name: "Max slider",
                default: 100,
            },
            /** Sets the starting value of the slider */
            slider_start: {
                type: ParameterType.INT,
                pretty_name: "Slider starting value",
                default: 50,
            },
            /** Sets the step of the slider */
            step: {
                type: ParameterType.INT,
                pretty_name: "Step",
                default: 1,
            },
            /** Array containing the labels for the slider. Labels will be displayed at equidistant locations along the slider. */
            labels: {
                type: ParameterType.HTML_STRING,
                pretty_name: "Labels",
                default: [],
                array: true,
            },
            /** Width of the slider in pixels. */
            slider_width: {
                type: ParameterType.INT,
                pretty_name: "Slider width",
                default: null,
            },
            /** Label of the button to advance. */
            button_label: {
                type: ParameterType.STRING,
                pretty_name: "Button label",
                default: "Continue",
                array: false,
            },
            /** If true, the participant will have to move the slider before continuing. */
            require_movement: {
                type: ParameterType.BOOL,
                pretty_name: "Require movement",
                default: false,
            },
            /** Any content here will be displayed below the slider. */
            prompts: {
                type: ParameterType.HTML_STRING,
                pretty_name: "Prompt",
                default: undefined,
                array: true,
            },
            /** How long to show the stimulus. */
            stimulus_duration: {
                type: ParameterType.INT,
                pretty_name: "Stimulus duration",
                default: null,
            },
            /** How long to show the trial. */
            trial_duration: {
                type: ParameterType.INT,
                pretty_name: "Trial duration",
                default: null,
            },
            /** If true, trial will end when user makes a response. */
            response_ends_trial: {
                type: ParameterType.BOOL,
                pretty_name: "Response ends trial",
                default: true,
            },
        },
    };

    /**
     * **html-slider-response**
     *
     * jsPsych plugin for showing an HTML stimulus and collecting a slider response
     *
     * @author Josh de Leeuw
     * @see {@link https://www.jspsych.org/plugins/jspsych-html-slider-response/ html-slider-response plugin documentation on jspsych.org}
     */
    class HtmlSliderResponsePlugin {
        static info = info;

        constructor(jsPsych) {
            this.jsPsych = jsPsych;
        }

        trial(display_element, trial) {
            // half of the thumb width value from jspsych.css, used to adjust the label positions
            var half_thumb_width = 7.5;

            var html = '<div id="jspsych-html-slider-response-wrapper" style="margin: 100px 0px;">';
            html += '<div id="jspsych-html-slider-response-stimulus">' + trial.stimulus + "</div>";
            html +=
                '<div class="jspsych-html-slider-response-container" style="position:relative; margin: 0 auto 3em auto; ';
            if (trial.slider_width !== null) {
                html += "width:" + trial.slider_width + "%;";
            } else {
                html += "width:auto;";
            }
            html += '">';

            for (let i = 0; i < trial.prompts.length; i++) {
                if (trial.prompts && trial.prompts[i] !== undefined) {
                    html += trial.prompts[i];
                }

                html += `<div class='jspsych-html-slider-response-group' id='jspsych-html-slider-response-group${i}'>`;

                for (let j = 0; j < trial.labels[i].length; j++) {
                    html += `<div class='jspsych-html-slider-response-container' id='jspsych-html-slider-response-container-group${i}-number${j}'>`;
                    html +=
                        '<input type="range" class="jspsych-slider" value="' +
                        trial.slider_start +
                        '" min="' +
                        trial.min +
                        '" max="' +
                        trial.max +
                        '" step="' +
                        trial.step +
                        '" group="' +
                        i +
                        '" number="' +
                        j +
                        `" id="jspsych-html-slider-response-response-group${i}-number${j}"></input>`;
                    // html += "</div>";

                    for (var k = 0; k < trial.labels[i][j].length; k++) {
                        var label_width_perc = 100 / (trial.labels[i][j].length - 1);
                        var percent_of_range = k * (100 / (trial.labels[i][j].length - 1));
                        var percent_dist_from_center = ((percent_of_range - 50) / 50) * 100;
                        var offset = (percent_dist_from_center * half_thumb_width) / 100;
                        html +=
                            '<div style="border: 1px solid transparent; display: inline-block; position: absolute; ' +
                            "left:calc(" +
                            percent_of_range +
                            "% - (" +
                            label_width_perc +
                            "% / 2) - " +
                            offset +
                            "px); text-align: center; width: " +
                            label_width_perc +
                            '%;">';
                        html +=
                            '<span style="text-align: center; font-size: 80%;">' +
                            trial.labels[i][j][k] +
                            "</span>";
                        html += "</div>";
                    }
                    html += "</div>";
                }

                html += "</div>";
            }
            html += "</div>";
            html += "</div>";
            html += "</div>";

            // add submit button
            html +=
                '<button id="jspsych-html-slider-response-next" class="jspsych-btn" ' +
                (trial.require_movement ? "disabled" : "") +
                ">" +
                trial.button_label +
                "</button>";

            display_element.innerHTML = html;

            var response = {
                rt: null,
                responses: Array(trial.labels.length)
                    .fill(null)
                    .map((_, i) => {
                        return Array(trial.labels[i].length).fill(null);
                    }),
            };

            console.log(response);

            if (trial.require_movement) {
                const enable_button = () => {
                    if (
                        response.responses.some((r) => {
                            return r.some((x) => x === null);
                        })
                    ) {
                        return;
                    }
                    display_element.querySelector("#jspsych-html-slider-response-next").disabled = false;
                };

                display_element.querySelectorAll(".jspsych-slider").forEach((el) => {
                    let group = parseInt(el.getAttribute("group"));
                    let number = parseInt(el.getAttribute("number"));

                    function update() {
                        let value = el.valueAsNumber;

                        response.responses[group][number] = value;

                        enable_button();
                    }

                    el.addEventListener("mousedown", (e) => {
                        update();
                    });

                    el.addEventListener("touchstart", (e) => {
                        update();
                    });

                    el.addEventListener("change", (e) => {
                        update();
                    });
                });
            }

            const end_trial = () => {
                this.jsPsych.pluginAPI.clearAllTimeouts();

                // save data
                var trialdata = {
                    rt: response.rt,
                    stimulus: trial.stimulus,
                    slider_start: trial.slider_start,
                    response: response.responses,
                    labels: trial.labels,
                    prompts: trial.prompts,
                };

                display_element.innerHTML = "";

                // next trial
                this.jsPsych.finishTrial(trialdata);
            };

            display_element
                .querySelector("#jspsych-html-slider-response-next")
                .addEventListener("click", () => {
                    // measure response time
                    var endTime = performance.now();
                    response.rt = Math.round(endTime - startTime);

                    if (trial.response_ends_trial) {
                        end_trial();
                    } else {
                        display_element.querySelector("#jspsych-html-slider-response-next").disabled = true;
                    }
                });

            if (trial.stimulus_duration !== null) {
                this.jsPsych.pluginAPI.setTimeout(() => {
                    display_element.querySelector("#jspsych-html-slider-response-stimulus").style.visibility =
                        "hidden";
                }, trial.stimulus_duration);
            }

            // end trial if trial_duration is set
            if (trial.trial_duration !== null) {
                this.jsPsych.pluginAPI.setTimeout(end_trial, trial.trial_duration);
            }

            var startTime = performance.now();
        }
    }

    return HtmlSliderResponsePlugin;
})(jsPsychModule);

// return jsPsychHtmlButtonSliderResponse;
