This document shows an example of using our sampler to perform the numerical correctness check shown in Appendix B.

```python
from sampler import *
```

First, we define the environment. In the $4\times4$ gridworld, states are Python tuples of $(x, y)$ locations. We define the "doors" (or possible entry points), the state prior over the doors (uniform), a method to randomly sample a start state according to that prior, as well as a transition function that iterates over neighbors of a state and costs to go to that neighbor. Finally, we give a heuristic for A\* search to use.

```python
N = 4
doors = [(0, i) for i in range(N)]
class basic_problem(classical_problem):
    def __init__(self):
        super().__init__(beta=2)

    def state_prior(self, x):
        return 1/len(doors) if x in doors else 0.

    def sample_start_state(self):
        return random.choice(doors)

    def neighbors(self, z):
        x, y = z
        yield z, 1
        if x + 1 < N:
            yield (x + 1, y), 1
        if x - 1 >= 0:
            yield (x - 1, y), 1
        if y + 1 < N:
            yield (x, y + 1), 1
        if y - 1 >= 0:
            yield (x, y - 1), 1

    def heuristic(self, z, end):
        return abs(z[0] - end[0]) + abs(z[1] - end[1])

problem = basic_problem()
```

Next, we use our sampler, as well as rejection sampling, to compute likelihood estimates for each state in the world.

```python
g = (N - 1, N - 1)
tasks = list(itertools.product(range(N), range(N)))
cache = compute_bdpt_cache(g, problem, num_samples=1000, depth=N)
out1 = []
out2 = []
out3 = []
for z in tqdm(tasks):
    size_principle = True
    out1.append(sample_likelihood(z, g, problem, num_samples=25000, depth=N, bdpt_cache=cache, size_principle=size_principle))
    out2.append(sample_likelihood_rejection(z, g, problem, num_samples=25000, size_principle=size_principle))
    out3.append(sample_likelihood(z, g, problem, num_samples=25000, depth=N, bdpt_cache=None, size_principle=size_principle))
out1 = np.array(out1)
out2 = np.array(out2)
out3 = np.array(out3)
```

Finally, we prepare the graph shown in the paper.

```python
plt.figure(figsize=(8, 4))

plt.subplot(1, 2, 1)
plt.bar(0.0 + np.arange(1, 1+len(out1)), out3 , align='edge', width=+0.3, label='Ours')
plt.bar(0.3 + np.arange(1, 1+len(out1)), out1 , align='edge', width=+0.3, label='Ours + BDPT')
plt.bar(0.6 + np.arange(1, 1+len(out1)), out2, align='edge', width=+0.3, label='Rejection')
plt.xticks(0.9/2 + np.arange(1, 1+len(out1)), [str(x) for x in tasks], rotation=90)
plt.xlabel('Observed location x')
plt.ylabel(f'Likelihood p(x | g=({N-1}, {N-1}))')
plt.legend()

plt.subplot(1, 2, 2)
plt.plot(out2, out3, '+', label='Ours')
plt.plot(out2, out1, 'x', label='Ours + BDPT')
plt.plot([0, np.max(out1)], [0, np.max(out1)], 'y:')
plt.xlabel('Likelihood (rejection)')
plt.ylabel('Likelihood (ours)')
plt.axis('equal')
plt.legend()

plt.suptitle('Our sampler is unbiased and precisely matches rejection sampling')
plt.tight_layout()

plt.savefig('out/sampler-correctness.pdf')
```