#!/usr/bin/env python3
# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
import av
import re
import os
import random

import cv2
import numpy as np
import csv
import torch
import torch.utils.data
from torchvision import transforms
from tqdm import tqdm

import slowfast.utils.logging as logging
from slowfast.utils.env import pathmgr

from . import decoder as decoder
from . import utils as utils
from . import video_container as container
from .build import DATASET_REGISTRY
from .random_erasing import RandomErasing
from .transform import create_random_augment
from .gaze_io_sample import parse_gtea_gaze

logger = logging.get_logger(__name__)


@DATASET_REGISTRY.register()
class Rlbenchgaze(torch.utils.data.Dataset):
    """
    RLBench video loader with gaze tracking data.
    Directory structure:
    - /path/to/future/
      ├── clips.gaze/ (short clips)
      ├── full_scale.gaze/ (full videos)
      ├── gaze/ (CSV annotations)
      └── gaze_frame_label/ (processed frame labels)
    """
    def __init__(self, cfg, mode, num_retries=10):
        # Only support train, val, and test mode.
        assert mode in ["train", "val", "test"], "Split '{}' not supported for Rlbenchgaze".format(mode)
        self.mode = mode
        self.cfg = cfg
        self._num_retries = num_retries
        self._video_meta = {}
        
        # Clip sampling configuration
        if self.mode in ["train", "val"]:
            self._num_clips = 1
        elif self.mode in ["test"]:
            self._num_clips = (cfg.TEST.NUM_ENSEMBLE_VIEWS * cfg.TEST.NUM_SPATIAL_CROPS)

        self.aug = False
        self.rand_erase = False

        if self.mode == "train" and cfg.AUG.ENABLE:  # use RandAug
            self.aug = True
            if self.cfg.AUG.RE_PROB > 0:
                self.rand_erase = True

        logger.info("Constructing Rlbenchgaze {}...".format(mode))
        self._construct_loader()
        self.use_temporal_gradient = False
        self.temporal_gradient_rate = 0.0

    def _construct_loader(self):
        """Build video loader based on directory structure"""
        if self.mode == 'train':
            path_to_file = 'data/train_rlbench_gaze.csv'
        elif self.mode == 'val' or self.mode == 'test':
            path_to_file = 'data/test_rlbench_gaze.csv'
        else:
            raise ValueError(f"Don't support mode {self.mode}.")

        assert pathmgr.exists(path_to_file), "{} dir not found".format(path_to_file)

        self._path_to_videos = []
        self._labels = dict()
        self._spatial_temporal_idx = []
        # Load video paths
        with pathmgr.open(path_to_file, "r") as f:
            paths = [item for item in f.read().splitlines()]
            for clip_idx, path in enumerate(paths):
                for idx in range(self._num_clips):
                    self._path_to_videos.append(os.path.join(self.cfg.DATA.PATH_PREFIX, path))
                    self._spatial_temporal_idx.append(idx)  # used in test
                    self._video_meta[clip_idx * self._num_clips + idx] = {}  # only used in torchvision backend
        assert (len(self._path_to_videos) > 0), "Failed to load Ego4dgaze split {} from {}".format(self._split_idx, path_to_file)

        for path in tqdm(self._path_to_videos):
            # 验证视频可读性
            try:
                with av.open(path) as container:
                    if len(container.streams.video) == 0:
                        logger.warning(f"No video stream found in {path}")
                        continue
            except Exception as e:
                logger.warning(f"Invalid video file {path}: {str(e)}")
                continue
        
        # Read gaze label
        logger.info('Loading Gaze Labels...')
        for path in tqdm(self._path_to_videos):
            # Extract task and episode info from path
            # Path format: .../clips.gaze/task/ep_X_stX/task_ep_X_stX_t0_t5.mp4
            parts = path.split('/')
            task = parts[-3]
            ep_file = parts[-1]

            # 使用正则表达式提取episode编号
            match = re.search(r'ep_(\d+)', ep_file)
            if not match:
                raise ValueError(f"无法从文件名提取episode编号: {ep_file}")
            ep_num = match.group(1)  # 返回 "9"
            
            # Construct label path
            for st in ["st1", "st2"]:
                label_name = f"ep_{ep_num}_{st}_frame_label.csv"
                label_path = os.path.join(
                    self.cfg.DATA.PATH_PREFIX,
                    "gaze_frame_label",
                    task,
                    label_name
                )

                if not pathmgr.exists(label_path):
                    logger.warning(f"Label file not found: {label_path}")
                    continue

                # Read label file
                with open(label_path, "r") as f:
                    rows = [list(map(float, row)) for i, row in enumerate(csv.reader(f)) if i > 0]
                
                # Store as numpy array [frame, x, y, gaze_type]
                self._labels[f"{task}_{ep_num}_{st}"] = np.array(rows)[:, 1:]  # Exclude frame index  # [x, y, type,] in line with egtea format

        assert len(self._path_to_videos) > 0, "No videos found"
        logger.info(f"Loaded {len(self._path_to_videos)} videos from {path_to_file}")
    
    def __getitem__(self, index):
        """
        Adapted for RLBench dataset structure with gaze tracking.
        Returns:
            frames (tensor): C x T x H x W
            gaze_label (tensor): T x 3 (x, y, type)
            gaze_heatmap (tensor): T x H/4 x W/4
            index (int): video index
            meta (dict): containing path and frame indices
        """
        if isinstance(index, tuple):
            index, short_cycle_idx = index
        else:
            short_cycle_idx = None

        if self.mode in ["train"]:
            # -1 indicates random sampling.
            temporal_sample_index = -1
            spatial_sample_index = -1
            min_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[0]  # 256
            max_scale = self.cfg.DATA.TRAIN_JITTER_SCALES[1]  # 320
            crop_size = self.cfg.DATA.TRAIN_CROP_SIZE  # 224
            if short_cycle_idx in [0, 1]:
                crop_size = int(round(self.cfg.MULTIGRID.SHORT_CYCLE_FACTORS[short_cycle_idx] * self.cfg.MULTIGRID.DEFAULT_S))
            if self.cfg.MULTIGRID.DEFAULT_S > 0:
                # Decreasing the scale is equivalent to using a larger "span"
                # in a sampling grid.
                min_scale = int(round(float(min_scale) * crop_size / self.cfg.MULTIGRID.DEFAULT_S))

        elif self.mode in ["val", "test"]:
            temporal_sample_index = (self._spatial_temporal_idx[index] // self.cfg.TEST.NUM_SPATIAL_CROPS)  # = 0
            # spatial_sample_index is in [0, 1, 2]. Corresponding to left,
            # center, or right if width is larger than height, and top, middle,
            # or bottom if height is larger than width.
            spatial_sample_index = ((self._spatial_temporal_idx[index] % self.cfg.TEST.NUM_SPATIAL_CROPS) if self.cfg.TEST.NUM_SPATIAL_CROPS > 1 else 1)  # = 1
            min_scale, max_scale, crop_size = (
                [self.cfg.DATA.TEST_CROP_SIZE] * 3
                # Don't understand why different scale is used when NUM_SPATIAL_CROPS>1
                # if self.cfg.TEST.NUM_SPATIAL_CROPS > 1
                # else [self.cfg.DATA.TRAIN_JITTER_SCALES[0]] * 2 + [self.cfg.DATA.TEST_CROP_SIZE]
            )  # = (256, 256, 256)
            # The testing is deterministic and no jitter should be performed.
            # min_scale, max_scale, and crop_size are expect to be the same.
            assert len({min_scale, max_scale}) == 1
        
        else:
            raise NotImplementedError("Does not support {} mode".format(self.mode))

        sampling_rate = utils.get_random_sampling_rate(self.cfg.MULTIGRID.LONG_CYCLE_SAMPLING_RATE, self.cfg.DATA.SAMPLING_RATE)
        # = 8

        # Try to decode and sample a clip from a video. If the video can not be
        # decoded, repeatedly find a random video replacement that can be decoded.
        for i_try in range(self._num_retries):

            video_path = self._path_to_videos[index]
            
            # video_container = None
            # try:
            #     video_container = container.get_video_container(
            #         self._path_to_videos[index],
            #         self.cfg.DATA_LOADER.ENABLE_MULTI_THREAD_DECODE,
            #         self.cfg.DATA.DECODING_BACKEND,
            #     )
            # except Exception as e:
            #     logger.info("Failed to load video from {} with error {}".format(self._path_to_videos[index], e))

            # Select a random video if the current video was not able to access.
            # if video_container is None:
            #     logger.warning("Failed to meta load video idx {} from {}; trial {}".format(index, self._path_to_videos[index], i_try))
            #     if self.mode not in ["test"] and i_try > self._num_retries // 2:
            #         # let's try another one
            #         index = random.randint(0, len(self._path_to_videos) - 1)
            #     continue
            
            with av.open(video_path, timeout=10) as video_container:
                try:
                    # 显式帧处理（避免隐式数组转换）
                    frames = []
                    for i, frame in enumerate(video_container.decode(video=0)):
                        if len(frames) >= self.cfg.DATA.NUM_FRAMES:
                            break
                        frame_data = frame.to_ndarray(format='rgb24')  # 明确指定格式
                        frames.append(np.asarray(frame_data))  # 显式转换
                    
                    if not frames:
                        raise ValueError("Empty video frames")
                        
                    frames = torch.from_numpy(np.stack(frames)).float()
                    
                    # 处理标签（显式索引转换）
                    frames_idx = np.arange(len(frames))
                    
                except Exception as e:
                    logger.error(f"Decode failed: {str(e)}")
                    video_container.close()

            # Decode video. Meta info is used to perform selective decoding.
            # frames, frames_idx = decoder.decode(
            #     container=video_container,
            #     sampling_rate=sampling_rate,
            #     num_frames=self.cfg.DATA.NUM_FRAMES,
            #     clip_idx=temporal_sample_index,
            #     num_clips=self.cfg.TEST.NUM_ENSEMBLE_VIEWS,
            #     video_meta=self._video_meta[index],
            #     target_fps=self.cfg.DATA.TARGET_FPS,
            #     backend=self.cfg.DATA.DECODING_BACKEND,
            #     max_spatial_scale=min_scale,  # only used in torchvision backend
            #     use_offset=self.cfg.DATA.USE_OFFSET_SAMPLING,
            #     get_frame_idx=True
            # )

            # Gaze label processing
            path_parts = video_path.split('/')
            task_name = path_parts[-3]  # .../clips.gaze/task/ep_X_stX/video.mp4
            ep_file = path_parts[-1]    # 文件名（如"open_drawer_ep_13_st1_t0_t3.5.mp4"）

            # 使用正则表达式提取episode编号和stage
            ep_match = re.search(r'ep_(\d+)_(st\d+)', ep_file)
            if not ep_match:
                raise ValueError(f"无法从文件名提取episode信息: {ep_file}")
                
            ep_num = ep_match.group(1)  # "13"
            stage = ep_match.group(2)   # "st1"

            video_name = f"{task_name}_{ep_num}_{stage}"  # 如"open_drawer_13_st1"
            if video_name not in self._labels:
                print(f"video_path = {video_path}, video_name = {video_name}")
                # video_path = /fs-computility/efm/shared/datasets/Official_Manipulation_Data/sim/colosseum/rlbench/future/
                # clips.gaze/slide_block_to_color_target/ep_99_st2/slide_block_to_color_target_ep_99_st2_t5_t6.9.mp4, 
                # video_name = slide_block_to_color_target_99
                raise KeyError(f"Gaze label not found for {video_name}")
            _, clip_name = video_path.split('/')[-2:]
            clip_tstart, clip_tend = clip_name[:-4].split('_')[-2:]  # get start and end time
            clip_tstart, clip_tend = float(clip_tstart[1:]), float(clip_tend[1:])  # remove 't'
            clip_fstart, clip_fend = clip_tstart * self.cfg.DATA.TARGET_FPS, clip_tend * self.cfg.DATA.TARGET_FPS

            # frames_global_idx = frames_idx.numpy() + clip_fstart - 1
            frames_global_idx = np.array(frames_idx) + clip_fstart - 1
            if self.mode not in ['test'] and frames_global_idx[-1] >= self._labels[video_name].shape[0]:  # Some frames don't have labels. Try to use another one
                # logger.info('No annotations:', video_name, clip_name)
                index = random.randint(0, len(self._path_to_videos) - 1)
                continue
            label = self._labels[video_name][frames_global_idx.astype(int), :] # 确保int类型

            # If decoding failed (wrong format, video is too short, and etc),
            # select another video.
            if frames is None:
                logger.warning("Failed to decode video idx {} from {}; trial {}".format(index, self._path_to_videos[index], i_try))
                if self.mode not in ["test"] and i_try > self._num_retries // 2:
                    # let's try another one
                    index = random.randint(0, len(self._path_to_videos) - 1)
                continue

            # Data augmentation
            if self.aug:
                if self.cfg.AUG.NUM_SAMPLE > 1:

                    frame_list = []
                    label_list = []
                    index_list = []
                    for _ in range(self.cfg.AUG.NUM_SAMPLE):
                        new_frames = self._aug_frame(frames, spatial_sample_index, min_scale, max_scale, crop_size)
                        label = self._labels[index]
                        new_frames = utils.pack_pathway_output(self.cfg, new_frames)
                        frame_list.append(new_frames)
                        label_list.append(label)
                        index_list.append(index)
                    return frame_list, label_list, index_list, {}

                else:
                    frames = self._aug_frame(frames, spatial_sample_index, min_scale, max_scale, crop_size)

            else:
                frames = utils.tensor_normalize(frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD)
                # T H W C -> C T H W.
                frames = frames.permute(3, 0, 1, 2)
                # Perform data augmentation.
                frames, label = utils.spatial_sampling(
                    frames,
                    gaze_loc=label, 
                    spatial_idx=spatial_sample_index,
                    min_scale=min_scale,
                    max_scale=max_scale,
                    crop_size=crop_size,
                    random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
                    inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
                )

            frames = utils.pack_pathway_output(self.cfg, frames)

            # Generate gaze heatmaps
            gaze_heatmaps = np.zeros(shape=(frames[0].size(1), frames[0].size(2) // 4, frames[0].size(3) // 4))
            for t in range(gaze_heatmaps.shape[0]):
                self._get_gaussian_map(
                    gaze_heatmaps[t],
                    center=(label[t, 0] * gaze_heatmaps.shape[2], label[t, 1] * gaze_heatmaps.shape[1]),
                    kernel_size=self.cfg.DATA.GAUSSIAN_KERNEL,
                    sigma=-1  # sigma=-1 means use default sigma
                )
                d_sum = gaze_heatmaps[t].sum()
                # Normalize
                if d_sum == 0:  # gaze may be outside the image
                    gaze_heatmaps[t] += 1 / (gaze_heatmaps.shape[1] * gaze_heatmaps.shape[2])
                elif d_sum != 1:  # gaze may be right at the edge of image
                    gaze_heatmaps[t] /= d_sum

            label_hm = torch.as_tensor(gaze_heatmaps).float()

            return frames, label, label_hm, index, {'path': self._path_to_videos[index], 'index': np.array(frames_global_idx), 
                                                    'task': task_name, 'episode': ep_num}
        else:
            raise RuntimeError("Failed to fetch video after {} retries.".format(self._num_retries))


    def _aug_frame(
        self,
        frames,
        spatial_sample_index,
        min_scale,
        max_scale,
        crop_size,
    ):
        aug_transform = create_random_augment(
            input_size=(frames.size(1), frames.size(2)),
            auto_augment=self.cfg.AUG.AA_TYPE,
            interpolation=self.cfg.AUG.INTERPOLATION,
        )
        # T H W C -> T C H W.
        frames = frames.permute(0, 3, 1, 2)
        list_img = self._frame_to_list_img(frames)
        list_img = aug_transform(list_img)
        frames = self._list_img_to_frames(list_img)
        frames = frames.permute(0, 2, 3, 1)

        frames = utils.tensor_normalize(frames, self.cfg.DATA.MEAN, self.cfg.DATA.STD)
        # T H W C -> C T H W.
        frames = frames.permute(3, 0, 1, 2)
        # Perform data augmentation.
        scl, asp = (self.cfg.DATA.TRAIN_JITTER_SCALES_RELATIVE, self.cfg.DATA.TRAIN_JITTER_ASPECT_RELATIVE)
        relative_scales = (None if (self.mode not in ["train"] or len(scl) == 0) else scl)
        relative_aspect = (None if (self.mode not in ["train"] or len(asp) == 0) else asp)
        frames = utils.spatial_sampling(
            frames,
            spatial_idx=spatial_sample_index,
            min_scale=min_scale,
            max_scale=max_scale,
            crop_size=crop_size,
            random_horizontal_flip=self.cfg.DATA.RANDOM_FLIP,
            inverse_uniform_sampling=self.cfg.DATA.INV_UNIFORM_SAMPLE,
            aspect_ratio=relative_aspect,
            scale=relative_scales,
            motion_shift=self.cfg.DATA.TRAIN_JITTER_MOTION_SHIFT
            if self.mode in ["train"]
            else False,
        )

        if self.rand_erase:
            erase_transform = RandomErasing(
                self.cfg.AUG.RE_PROB,
                mode=self.cfg.AUG.RE_MODE,
                max_count=self.cfg.AUG.RE_COUNT,
                num_splits=self.cfg.AUG.RE_COUNT,
                device="cpu",
            )
            frames = frames.permute(1, 0, 2, 3)
            frames = erase_transform(frames)
            frames = frames.permute(1, 0, 2, 3)

        return frames
    
    def _frame_to_list_img(self, frames):
        img_list = [transforms.ToPILImage()(frames[i]) for i in range(frames.size(0))]
        return img_list

    def _list_img_to_frames(self, img_list):
        img_list = [transforms.ToTensor()(img) for img in img_list]
        return torch.stack(img_list)

    @staticmethod
    def _get_gaussian_map(heatmap, center, kernel_size, sigma):
        h, w = heatmap.shape
        mu_x, mu_y = round(center[0]), round(center[1])
        left = max(mu_x - (kernel_size - 1) // 2, 0)
        right = min(mu_x + (kernel_size - 1) // 2, w-1)
        top = max(mu_y - (kernel_size - 1) // 2, 0)
        bottom = min(mu_y + (kernel_size - 1) // 2, h-1)

        if left >= right or top >= bottom:
            pass
        else:
            kernel_1d = cv2.getGaussianKernel(ksize=kernel_size, sigma=sigma, ktype=cv2.CV_32F)
            kernel_2d = kernel_1d * kernel_1d.T
            k_left = (kernel_size - 1) // 2 - mu_x + left
            k_right = (kernel_size - 1) // 2 + right - mu_x
            k_top = (kernel_size - 1) // 2 - mu_y + top
            k_bottom = (kernel_size - 1) // 2 + bottom - mu_y

            heatmap[top:bottom+1, left:right+1] = kernel_2d[k_top:k_bottom+1, k_left:k_right+1]

    def __len__(self):
        """
        Returns:
            (int): the number of videos in the dataset.
        """
        return self.num_videos

    @property
    def num_videos(self):
        """
        Returns:
            (int): the number of videos in the dataset.
        """
        return len(self._path_to_videos)
