# coding=utf-8
# Copyright 2022 The Mixed Fl Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for classifier_model."""

from absl.testing import parameterized
import tensorflow as tf

from mixed_fl.experiments.emnist import classifier_model

BATCH_SIZE = 5
EMNIST_SIDE_LENGTH = 28


def _get_batched_image():
  return tf.random.uniform(
      shape=[BATCH_SIZE, EMNIST_SIDE_LENGTH, EMNIST_SIDE_LENGTH, 1], maxval=1.0,
      minval=-1.0, dtype=tf.float32)


class ClassifierModelTest(tf.test.TestCase, parameterized.TestCase):

  def test_get_emnist_classifier_model_input_spec(self):
    model = classifier_model.get_emnist_classifier_model()
    self.assertLen(model.input_spec, 1)
    input_spec = model.input_spec[0]
    self.assertEqual(input_spec.shape,
                     (None, EMNIST_SIDE_LENGTH, EMNIST_SIDE_LENGTH, 1))
    self.assertEqual(input_spec.ndim, 4)

  @parameterized.named_parameters(('with_10_classes', 10),
                                  ('with_36_classes', 36),
                                  ('with_62_classes', 36))
  def test_get_emnist_classifier_model_works(self, num_classes):
    model = classifier_model.get_emnist_classifier_model(num_classes)
    batched_image = _get_batched_image()
    output = model(batched_image)
    self.assertEqual(output.shape, [BATCH_SIZE, num_classes])
    self.assertDTypeEqual(output, tf.float32)


if __name__ == '__main__':
  tf.test.main()
