# coding=utf-8
# Copyright 2022 The Mixed Fl Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for datasets."""

import tensorflow as tf
import tensorflow_federated as tff

from mixed_fl.experiments.celeba import datasets


class DatasetsLibTest(tf.test.TestCase):

  def test__merge_three_clients_into_one_works(self):
    unmerged_client_data = tff.simulation.datasets.TestClientData({
        '0': [0, 0, 0],
        '1': [1, 1, 1],
        '2': [2, 2, 2],
    })
    expected_client_id = '0_1_2'

    merged_client_data = datasets._merge_three_clients_into_one(
        unmerged_client_data)

    self.assertLen(merged_client_data.client_ids, 1)
    self.assertEqual([expected_client_id], merged_client_data.client_ids)

    self.assertCountEqual([0, 0, 0, 1, 1, 1, 2, 2, 2],
                          list(
                              merged_client_data.create_tf_dataset_for_client(
                                  expected_client_id).as_numpy_iterator()))

  def test__merge_three_clients_into_one_truncates_to_multiples_of_three(self):
    unmerged_client_data = tff.simulation.datasets.TestClientData({
        '0': [0, 0, 0],
        '1': [1, 1, 1],
        '2': [2, 2, 2],
        '3': [3, 3, 3],
        '4': [4, 4, 4],
        '5': [5, 5, 5],
        # This unmerged client will get left out of the merged clients and data.
        '6': [6, 6, 6],
    })
    expected_client_ids = ['0_1_2', '3_4_5']

    merged_client_data = datasets._merge_three_clients_into_one(
        unmerged_client_data)

    self.assertLen(merged_client_data.client_ids, 2)
    self.assertEqual(expected_client_ids, merged_client_data.client_ids)

    self.assertCountEqual([0, 0, 0, 1, 1, 1, 2, 2, 2],
                          list(
                              merged_client_data.create_tf_dataset_for_client(
                                  expected_client_ids[0]).as_numpy_iterator()))
    self.assertCountEqual([3, 3, 3, 4, 4, 4, 5, 5, 5],
                          list(
                              merged_client_data.create_tf_dataset_for_client(
                                  expected_client_ids[1]).as_numpy_iterator()))

  def test__merge_three_clients_into_one_raises_error_with_two_clients(self):
    unmerged_client_data = tff.simulation.datasets.TestClientData({
        '0': [0, 0, 0],
        '1': [1, 1, 1],
    })

    with self.assertRaises(ValueError):
      datasets._merge_three_clients_into_one(unmerged_client_data)


if __name__ == '__main__':
  tf.test.main()
