# coding=utf-8
# Copyright 2022 The Mixed Fl Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for prediction_model."""

import tensorflow as tf

from mixed_fl.experiments.next_char_prediction import prediction_model

BATCH_SIZE = 7
SEQ_LENGTH = 42


def _get_batched_preprocessed_text_sequence():
  # In preprocessing, the text is split into characters, the characters are
  # mapped to corresponding integers, and then individual examples are batched.
  # This random batch represents the output of that process.
  return tf.random.uniform(
      shape=[BATCH_SIZE, SEQ_LENGTH],
      maxval=prediction_model.VOCAB_SIZE,
      minval=0,
      dtype=tf.int64)


class PredictionModelTest(tf.test.TestCase):

  def test_get_next_char_prediction_model_input_spec(self):
    model = prediction_model.get_next_char_prediction_model()
    self.assertLen(model.input_spec, 1)
    input_spec = model.input_spec[0]
    self.assertEqual(input_spec.shape, (None, None))
    self.assertEqual(input_spec.ndim, 2)

  def test_get_next_char_prediction_model_works(self):
    model = prediction_model.get_next_char_prediction_model()
    batched_text_sequence = _get_batched_preprocessed_text_sequence()
    output = model(batched_text_sequence)
    self.assertEqual(output.shape,
                     [BATCH_SIZE, SEQ_LENGTH, prediction_model.VOCAB_SIZE])
    self.assertDTypeEqual(output, tf.float32)


if __name__ == '__main__':
  tf.test.main()
