# https://github.com/lucidrains/vit-pytorch/blob/main/vit_pytorch/vit_pytorch.py

import torch
import torch.nn.functional as F
# from einops import rearrange
from torch import nn
import pdb

import copy
from utils.conf import get_device

MIN_NUM_PATCHES = 16

def APDLinear(x, key_weights, task_id, dims=[], module_key='', info='weights', tad_thr=0, device='cpu'):
    key = module_key + '__' + info
    tsh = get_key_weights(key_weights, key, 'tsh.shared', dims).to(device)
    msk = get_key_weights(key_weights, key, 'msk.t%d'%task_id, [dims[0]]).to(device)
    tad = get_key_weights(key_weights, key, 'tad.t%d'%task_id, dims, is_tad=True).to(device)
    #print(x.shape, tsh.shape, msk.shape, tad.shape)
    return F.linear(x, torch.einsum('i,i...->i...', msk, tsh) + l1_pruning(tad, tad_thr))

def l1_pruning(input, l1_hyp = 0):
    if l1_hyp == 0:
        return input
    else:
        hard_threshold = torch.abs(input) > l1_hyp
        return input * hard_threshold

def get_key_weights(key_weights, w_info, w_type, shape, is_tad=False):
    weights_type, weights_tid = w_type.split('.')
    key = '%s_%s'%(w_info, weights_tid)
    # get key w/o task_index
    if key in key_weights[weights_type].keys():
        return key_weights[weights_type][key]
    else:
        if not is_tad:
            w = nn.Parameter(torch.ones(shape, requires_grad=True))
            # initialization of mask: ones, otherwise: kaiming_normal
            nn.init.kaiming_normal_(w.data, mode='fan_out', nonlinearity='relu') if len(shape)>2 else None
            key_weights[weights_type][key] = w
        else:
            tsh_key = '%s_%s'%(w_info, 'shared')
            key_weights[weights_type][key] = copy.deepcopy(key_weights['tsh'][tsh_key])
            #key_weights[key] = copy.deepcopy(key_weights[tsh_key])
        return key_weights[weights_type][key]

class Residual(nn.Module):
    def __init__(self, fn):
        super().__init__()
        self.fn = fn
    def forward(self, x, **kwargs):
        return self.fn(x, **kwargs) + x

class PreNorm(nn.Module):
    def __init__(self, dim, fn):
        super().__init__()
        self.norm = nn.LayerNorm(dim)
        self.fn = fn
    def forward(self, x, **kwargs):
        return self.fn(self.norm(x), **kwargs)

class FeedForward(nn.Module):
    def __init__(self, dim, hidden_dim, dropout = 0., key_weights=None, tad_thr=0, device='cuda'):
        super().__init__()
        self.dim = dim
        self.hidden_dim = hidden_dim
        self.gelu = nn.GELU()
        self.dropout = nn.Dropout(dropout)
        self.key_weights = key_weights
        self.tad_thr = tad_thr
        self.device = device
        """
        self.net = nn.Sequential(
            nn.Linear(dim, hidden_dim),
            nn.GELU(),
            nn.Dropout(dropout),
            nn.Linear(hidden_dim, dim),
            nn.Dropout(dropout)
        )
        """

    def forward(self, x, task_id=0):
        out = APDLinear(x, self.key_weights, task_id, dims=[self.hidden_dim, self.dim],
                                        module_key='FFN', info='weights1',
                                        tad_thr=self.tad_thr, device=self.device)
        out = self.dropout(self.gelu(out))
        out = APDLinear(out, self.key_weights, task_id, dims=[self.dim, self.hidden_dim],
                                        module_key='FFN', info='weights2',
                                        tad_thr=self.tad_thr, device=self.device)
        return self.dropout(out)

class Attention(nn.Module):
    def __init__(self, dim, heads = 8, dropout = 0., key_weights = None, tad_thr=0, device='cuda'):
        super().__init__()
        self.heads = heads
        self.scale = dim ** -0.5
        self.dim = dim
        self.dropout = nn.Dropout(dropout)
        self.key_weights = key_weights
        self.tad_thr = tad_thr
        self.device = device
        """
        self.to_qkv = nn.Linear(dim, dim * 3, bias = False)
        self.to_out = nn.Sequential(
            nn.Linear(dim, dim),
            nn.Dropout(dropout)
        )
        """

    def forward(self, x, task_id=0, mask = None):
        b, n, _, h = *x.shape, self.heads
        #qkv = self.to_qkv(x).chunk(3, dim = -1)
        qkv = APDLinear(x, self.key_weights, task_id, dims=[self.dim * 3, self.dim],
                                        module_key='to_qkv', info='weights',
                                        tad_thr=self.tad_thr, device=self.device)
        qkv = qkv.chunk(3, dim = -1)
        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h = h), qkv)

        dots = torch.einsum('bhid,bhjd->bhij', q, k) * self.scale

        if mask is not None:
            mask = F.pad(mask.flatten(1), (1, 0), value = True)
            assert mask.shape[-1] == dots.shape[-1], 'mask has incorrect dimensions'
            mask = mask[:, None, :] * mask[:, :, None]
            dots.masked_fill_(~mask, float('-inf'))
            del mask

        attn = dots.softmax(dim=-1)

        out = torch.einsum('bhij,bhjd->bhid', attn, v)
        out = rearrange(out, 'b h n d -> b n (h d)')
        #out =  self.to_out(out)
        out = APDLinear(out, self.key_weights, task_id, dims=[self.dim, self.dim],
                                        module_key='to_out', info='weights',
                                        tad_thr=self.tad_thr, device=self.device)
        out = self.dropout(out)
        return out

class Transformer(nn.Module):
    def __init__(self, dim, depth, heads, mlp_dim, dropout, key_weights):
        super().__init__()
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                Residual(PreNorm(dim, Attention(dim, heads = heads, dropout = dropout, key_weights=key_weights))),
                Residual(PreNorm(dim, FeedForward(dim, mlp_dim, dropout = dropout, key_weights=key_weights)))
            ]))
    def forward(self, x, task_id, mask = None):
        for attn, ff in self.layers:
            pdb.set_trace()
            x = attn(x, task_id, mask = mask)
            x = ff(x, task_id)
        return x

class ViT(nn.Module):
    def __init__(self, *, image_size, patch_size, num_classes, dim, depth, heads, mlp_dim, channels = 3, dropout = 0., emb_dropout = 0., tad_thr=0):
        super().__init__()
        assert image_size % patch_size == 0, 'image dimensions must be divisible by the patch size'
        num_patches = (image_size // patch_size) ** 2
        patch_dim = channels * patch_size ** 2
        assert num_patches > MIN_NUM_PATCHES, f'your number of patches ({num_patches}) is way too small for attention to be effective. try decreasing your patch size'

        self.patch_size = patch_size
        self.patch_dim = patch_dim
        self.dim = dim
        self.device = get_device()
        self.tad_thr = tad_thr
        self.key_weights = {'tsh':nn.ParameterDict(),
                            'msk':nn.ParameterDict(),
                            'tad':nn.ParameterDict(),
                            'tmp_sol':{}, # Freezed
                            }

        self.pos_embedding = nn.Parameter(torch.randn(1, num_patches + 1, dim))
        #self.patch_to_embedding = nn.Linear(patch_dim, dim)
        self.cls_token = nn.Parameter(torch.randn(1, 1, dim))
        self.dropout = nn.Dropout(emb_dropout)

        self.transformer = Transformer(dim, depth, heads, mlp_dim, dropout, key_weights=self.key_weights)

        self.to_cls_token = nn.Identity()

        self.mlp_head = nn.Sequential(
            nn.LayerNorm(dim),
            nn.Linear(dim, mlp_dim),
            nn.GELU(),
            nn.Dropout(dropout),
            nn.Linear(mlp_dim, num_classes)
        )

    def forward(self, img, task_id, mask = None):
        p = self.patch_size

        x = rearrange(img, 'b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1 = p, p2 = p)
        #x = self.patch_to_embedding(x)
        x = APDLinear(x, self.key_weights, task_id, dims=[self.dim, self.patch_dim],
                        module_key='patch_to_embedding', info='weights', tad_thr=self.tad_thr, device=self.device)
        b, n, _ = x.shape
        cls_tokens = self.cls_token.expand(b, -1, -1)
        x = torch.cat((cls_tokens, x), dim=1)
        x += self.pos_embedding[:, :(n + 1)]
        x = self.dropout(x)

        x = self.transformer(x, task_id, mask=mask)

        x = self.to_cls_token(x[:, 0])
        return self.mlp_head(x)

    def get_params(self):
        params = []
        for pp in list(self.parameters()):
          # if pp.grad is not None:
          params.append(pp.view(-1))
        return torch.cat(params)

    def get_grads(self):
        grads = []
        for pp in list(self.parameters()):
            # if pp.grad is not None:
            grads.append(pp.grad.view(-1))
        return torch.cat(grads)
