from functools import partial

import torch
import torch.nn as nn
# import torch.nn.functional as F
# from timm.models.layers import trunc_normal_

# from .VisionTransformer import VisionTransformer

# MAE
from timm.models.vision_transformer import PatchEmbed, Block
from models.backbones.utils.pos_embed import get_2d_sincos_pos_embed

# class VisionTransformerForMAE(VisionTransformer):
#     def __init__(self, **kwargs):
#         super().__init__(**kwargs)

#         assert self.num_classes == 0

#         self.mask_token = nn.Parameter(torch.zeros(1, 1, self.embed_dim))
#         self._trunc_normal_(self.mask_token, std=.02)

#     def _trunc_normal_(self, tensor, mean=0., std=1.):
#         trunc_normal_(tensor, mean=mean, std=std, a=-std, b=std)

#     def forward(self, x, mask):
#         x = self.patch_embed(x)

#         assert mask is not None
#         B, L, _ = x.shape

#         mask_token = self.mask_token.expand(B, L, -1)
#         w = mask.flatten(1).unsqueeze(-1).type_as(mask_token)
#         x = x * (1 - w) + mask_token * w

#         cls_tokens = self.cls_token.expand(B, -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
#         x = torch.cat((cls_tokens, x), dim=1)

#         if self.pos_embed is not None:
#             x = x + self.pos_embed
#         x = self.pos_drop(x)

#         rel_pos_bias = self.rel_pos_bias() if self.rel_pos_bias is not None else None
#         for blk in self.blocks:
#             x = blk(x, rel_pos_bias=rel_pos_bias)
#         x = self.norm(x)

#         x = x[:, 1:]
#         B, L, C = x.shape
#         H = W = int(L ** 0.5)
#         x = x.permute(0, 2, 1).reshape(B, C, H, W).contiguous()
#         return x

# class MAE(nn.Module):
#     def __init__(self, encoder, encoder_stride):
#         super().__init__()
#         self.encoder = encoder
#         self.encoder_stride = encoder_stride

#         self.decoder = nn.Sequential(
#             nn.Conv2d(
#                 in_channels=self.encoder.num_features,
#                 out_channels=self.encoder_stride ** 2 * 3, kernel_size=1),
#             nn.PixelShuffle(self.encoder_stride),
#         )

#         self.in_chans = self.encoder.in_chans
#         self.patch_size = self.encoder.patch_size

#     def forward(self, x, mask, reduction='mean', return_recon=False, return_emb=False, colorwise=False):
#         z = self.encoder(x, mask)
#         x_rec = self.decoder(z)

#         mask = mask.repeat_interleave(self.patch_size, 1).repeat_interleave(self.patch_size, 2).unsqueeze(1).contiguous()
#         loss_recon = F.l1_loss(x, x_rec, reduction='none')
#         if reduction == 'mean':
#             loss = (loss_recon * mask).sum() / (mask.sum() + 1e-5) / self.in_chans
#         else:
#             loss = loss_recon               

#         if return_recon:
#             return loss, x_rec
#         elif return_emb:
#             return loss, z        
#         else:
#             return loss    

#     @torch.jit.ignore
#     def no_weight_decay(self):
#         if hasattr(self.encoder, 'no_weight_decay'):
#             return {'encoder.' + i for i in self.encoder.no_weight_decay()}
#         return {}

#     @torch.jit.ignore
#     def no_weight_decay_keywords(self):
#         if hasattr(self.encoder, 'no_weight_decay_keywords'):
#             return {'encoder.' + i for i in self.encoder.no_weight_decay_keywords()}
#         return {}


# def build_mae(**kwargs):
#     # if config.model.type == 'vit':
#     # model = VisionTransformerForMAE(
#     #     img_size=config.dataset.image_size,
#     #     patch_size=config.model.vit.patch_size,
#     #     in_chans=config.model.vit.in_chans,
#     #     # num_classes=0,
#     #     embed_dim=config.model.vit.embed_dim,
#     #     depth=config.model.vit.depth,
#     #     num_heads=config.model.vit.num_heads,
#     #     mlp_ratio=config.model.vit.mlp_ratio,
#     #     # qkv_bias=config.model.vit.qkv_bias,
#     #     # drop_rate=config.model.vit.drop_rate,
#     #     # drop_path_rate=config.model.vit.drop_path_rate,
#     #     norm_layer=partial(nn.LayerNorm, eps=1e-6),
#     #     # init_values=config.model.vit.init_values,
#     #     # use_abs_pos_emb=config.model.vit.use_ape,
#     #     # use_rel_pos_bias=config.model.vit.use_rpb,
#     #     # use_shared_rel_pos_bias=config.model.vit.use_shared_rpb,
#     #     # use_mean_pooling=config.model.vit.use_mean_pooling,            
#     #     decoder_embed_dim=512,
#     #     decoder_depth=8, 
#     #     decoder_num_heads=16,
#     #     norm_pix_loss=False)
#     # model = MAE(encoder=encoder, encoder_stride=encoder_stride)
#     # return model
#     model = MaskedAutoencoderViT(
#         patch_size=16, embed_dim=768, depth=12, num_heads=12,
#         decoder_embed_dim=512, decoder_depth=8, decoder_num_heads=16,
#         mlp_ratio=4, norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
#     return model
    
class MaskedAutoencoderViT(nn.Module):
    """ Masked Autoencoder with VisionTransformer backbone
    """    
# class MaskedAutoencoderViT(VisionTransformer):
    def __init__(self, img_size=224, patch_size=16, in_chans=3,
                 embed_dim=1024, depth=24, num_heads=16,
                 decoder_embed_dim=512, decoder_depth=8, decoder_num_heads=16,
                 mlp_ratio=4., norm_layer=nn.LayerNorm, norm_pix_loss=False):
#         super().__init__(img_size=224, patch_size=16, in_chans=3,
#                  embed_dim=1024, depth=24, num_heads=16,
#                  decoder_embed_dim=512, decoder_depth=8, decoder_num_heads=16,
#                  mlp_ratio=4., norm_layer=nn.LayerNorm, norm_pix_loss=False)
        super().__init__()

        # --------------------------------------------------------------------------
        # MAE encoder specifics           
        self.patch_embed = PatchEmbed(img_size, patch_size, in_chans, embed_dim)
        num_patches = self.patch_embed.num_patches

        self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim), requires_grad=False)  # fixed sin-cos embedding

        self.blocks = nn.ModuleList([
            # Block(embed_dim, num_heads, mlp_ratio, qkv_bias=True, qk_scale=None, norm_layer=norm_layer)
            Block(embed_dim, num_heads, mlp_ratio, qkv_bias=True, norm_layer=norm_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)
        # --------------------------------------------------------------------------

        # --------------------------------------------------------------------------
        # MAE decoder specifics
        self.decoder_embed = nn.Linear(embed_dim, decoder_embed_dim, bias=True)

        self.mask_token = nn.Parameter(torch.zeros(1, 1, decoder_embed_dim))

        self.decoder_pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, decoder_embed_dim), requires_grad=False)  # fixed sin-cos embedding

        self.decoder_blocks = nn.ModuleList([
            # Block(decoder_embed_dim, decoder_num_heads, mlp_ratio, qkv_bias=True, qk_scale=None, norm_layer=norm_layer)
            Block(decoder_embed_dim, decoder_num_heads, mlp_ratio, qkv_bias=True, norm_layer=norm_layer)
            for i in range(decoder_depth)])

        self.decoder_norm = norm_layer(decoder_embed_dim)
        self.decoder_pred = nn.Linear(decoder_embed_dim, patch_size**2 * in_chans, bias=True) # decoder to patch
        # --------------------------------------------------------------------------

        self.norm_pix_loss = norm_pix_loss

        self.initialize_weights()

    def initialize_weights(self):
        # initialization
        # initialize (and freeze) pos_embed by sin-cos embedding
        pos_embed = get_2d_sincos_pos_embed(self.pos_embed.shape[-1], int(self.patch_embed.num_patches**.5), cls_token=True)
        self.pos_embed.data.copy_(torch.from_numpy(pos_embed).float().unsqueeze(0))

        decoder_pos_embed = get_2d_sincos_pos_embed(self.decoder_pos_embed.shape[-1], int(self.patch_embed.num_patches**.5), cls_token=True)
        self.decoder_pos_embed.data.copy_(torch.from_numpy(decoder_pos_embed).float().unsqueeze(0))

        # initialize patch_embed like nn.Linear (instead of nn.Conv2d)
        w = self.patch_embed.proj.weight.data
        torch.nn.init.xavier_uniform_(w.view([w.shape[0], -1]))

        # timm's trunc_normal_(std=.02) is effectively normal_(std=0.02) as cutoff is too big (2.)
        torch.nn.init.normal_(self.cls_token, std=.02)
        torch.nn.init.normal_(self.mask_token, std=.02)

        # initialize nn.Linear and nn.LayerNorm
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            # we use xavier_uniform following official JAX ViT:
            torch.nn.init.xavier_uniform_(m.weight)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def patchify(self, imgs):
        """
        imgs: (N, 3, H, W)
        x: (N, L, patch_size**2 *3)
        """
        p = self.patch_embed.patch_size[0]
        assert imgs.shape[2] == imgs.shape[3] and imgs.shape[2] % p == 0

        h = w = imgs.shape[2] // p
        x = imgs.reshape(shape=(imgs.shape[0], 3, h, p, w, p))
        x = torch.einsum('nchpwq->nhwpqc', x)
        x = x.reshape(shape=(imgs.shape[0], h * w, p**2 * 3))
        return x

    def unpatchify(self, x):
        """
        x: (N, L, patch_size**2 *3)
        imgs: (N, 3, H, W)
        """
        p = self.patch_embed.patch_size[0]
        h = w = int(x.shape[1]**.5)
        assert h * w == x.shape[1]
        
        x = x.reshape(shape=(x.shape[0], h, w, p, p, 3))
        x = torch.einsum('nhwpqc->nchpwq', x)
        imgs = x.reshape(shape=(x.shape[0], 3, h * p, h * p))
        return imgs

    def random_masking(self, x, mask_ratio):
        """
        Perform per-sample random masking by per-sample shuffling.
        Per-sample shuffling is done by argsort random noise.
        x: [N, L, D], sequence
        """
        N, L, D = x.shape  # batch, length, dim
        len_keep = int(L * (1 - mask_ratio))
        
        noise = torch.rand(N, L, device=x.device)  # noise in [0, 1]
        
        # sort noise for each sample
        ids_shuffle = torch.argsort(noise, dim=1)  # ascend: small is keep, large is remove
        ids_restore = torch.argsort(ids_shuffle, dim=1)

        # keep the first subset
        ids_keep = ids_shuffle[:, :len_keep]
        x_masked = torch.gather(x, dim=1, index=ids_keep.unsqueeze(-1).repeat(1, 1, D))

        # generate the binary mask: 0 is keep, 1 is remove
        mask = torch.ones([N, L], device=x.device)
        mask[:, :len_keep] = 0
        # unshuffle to get the binary mask
        mask = torch.gather(mask, dim=1, index=ids_restore)

        return x_masked, mask, ids_restore

    def forward_encoder(self, x, mask_ratio):
        # embed patches
        x = self.patch_embed(x)

        # add pos embed w/o cls token
        x = x + self.pos_embed[:, 1:, :]

        # masking: length -> length * mask_ratio
        x, mask, ids_restore = self.random_masking(x, mask_ratio)

        # append cls token
        cls_token = self.cls_token + self.pos_embed[:, :1, :]
        cls_tokens = cls_token.expand(x.shape[0], -1, -1)
        x = torch.cat((cls_tokens, x), dim=1)

        # apply Transformer blocks
        for blk in self.blocks:
            x = blk(x)
        x = self.norm(x)

        return x, mask, ids_restore

    def forward_decoder(self, x, ids_restore):
        # embed tokens
        x = self.decoder_embed(x)

        # append mask tokens to sequence
        mask_tokens = self.mask_token.repeat(x.shape[0], ids_restore.shape[1] + 1 - x.shape[1], 1)
        x_ = torch.cat([x[:, 1:, :], mask_tokens], dim=1)  # no cls token
        x_ = torch.gather(x_, dim=1, index=ids_restore.unsqueeze(-1).repeat(1, 1, x.shape[2]))  # unshuffle
        x = torch.cat([x[:, :1, :], x_], dim=1)  # append cls token

        # add pos embed
        x = x + self.decoder_pos_embed

        # apply Transformer blocks
        for blk in self.decoder_blocks:
            x = blk(x)
        x = self.decoder_norm(x)

        # predictor projection
        x = self.decoder_pred(x)

        # remove cls token
        x = x[:, 1:, :]

        return x

    def forward_loss(self, imgs, pred, mask):
        """
        imgs: [N, 3, H, W]
        pred: [N, L, p*p*3]
        mask: [N, L], 0 is keep, 1 is remove, 
        """
        target = self.patchify(imgs)
        if self.norm_pix_loss:
            mean = target.mean(dim=-1, keepdim=True)
            var = target.var(dim=-1, keepdim=True)
            target = (target - mean) / (var + 1.e-6)**.5

        loss = (pred - target) ** 2
        loss = loss.mean(dim=-1)  # [N, L], mean loss per patch

        loss = (loss * mask).sum() / mask.sum()  # mean loss on removed patches
        return loss

    def forward(self, imgs, mask_ratio=0.75, reduction='mean', return_recon=False, return_emb=False):
        latent, mask, ids_restore = self.forward_encoder(imgs, mask_ratio)
        pred = self.forward_decoder(latent, ids_restore)  # [N, L, p*p*3]
        loss = self.forward_loss(imgs, pred, mask)
        # return loss, pred, mask
        return loss


    # def forward(self, x, mask, reduction='mean', return_recon=False, return_emb=False, colorwise=False):
    #     z = self.encoder(x, mask)
    #     x_rec = self.decoder(z)

    #     mask = mask.repeat_interleave(self.patch_size, 1).repeat_interleave(self.patch_size, 2).unsqueeze(1).contiguous()
    #     loss_recon = F.l1_loss(x, x_rec, reduction='none')
    #     if reduction == 'mean':
    #         loss = (loss_recon * mask).sum() / (mask.sum() + 1e-5) / self.in_chans
    #     else:
    #         loss = loss_recon               

    #     if return_recon:
    #         return loss, x_rec
    #     elif return_emb:
    #         return loss, z        
    #     else:
    #         return loss    

def build_mae(**kwargs):
    # decoder: 512 dim, 8 blocks
    model = MaskedAutoencoderViT(
        patch_size=16, embed_dim=768, depth=12, num_heads=12,
        decoder_embed_dim=512, decoder_depth=8, decoder_num_heads=16,
        mlp_ratio=4, norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    return model


# set recommended archs
# mae_vit_base_patch16 = mae_vit_base_patch16_dec512d8b  # decoder: 512 dim, 8 blocks