import torch
import torch.nn as nn
import torch.nn.functional as F
# from torchvision.models import resnet50
# from torchvision.models import resnet18
# from .ResNet import resnet50 as resnet50
from . import resnet50
from . import resnet18
from . import swin
from . import vit
from functools import partial


def D(p, z, version='simplified'): # negative cosine similarity
    if version == 'original':
        z = z.detach() # stop gradient
        p = F.normalize(p, dim=1) # l2-normalize
        z = F.normalize(z, dim=1) # l2-normalize
        return -(p*z).sum(dim=1).mean()

    elif version == 'simplified':# same thing, much faster. Scroll down, speed test in __main__
        return - F.cosine_similarity(p, z.detach(), dim=-1).mean()
    else:
        raise Exception



class projection_MLP(nn.Module):
    def __init__(self, in_dim, hidden_dim=2048, out_dim=2048):
        super().__init__()
        ''' page 3 baseline setting
        Projection MLP. The projection MLP (in f) has BN ap-
        plied to each fully-connected (fc) layer, including its out-
        put fc. Its output fc has no ReLU. The hidden fc is 2048-d.
        This MLP has 3 layers.
        '''
        self.layer1 = nn.Sequential(
            nn.Linear(in_dim, hidden_dim),
            nn.BatchNorm1d(hidden_dim),
            nn.ReLU(inplace=True)
        )
        self.layer2 = nn.Sequential(
            nn.Linear(hidden_dim, hidden_dim),
            nn.BatchNorm1d(hidden_dim),
            nn.ReLU(inplace=True)
        )
        self.layer3 = nn.Sequential(
            nn.Linear(hidden_dim, out_dim),
            nn.BatchNorm1d(hidden_dim)
        )
        self.num_layers = 2
    def set_layers(self, num_layers):
        self.num_layers = num_layers

    def forward(self, x):
        if self.num_layers == 3:
            x = self.layer1(x)
            x = self.layer2(x)
            x = self.layer3(x)
        elif self.num_layers == 2:
            x = self.layer1(x)
            x = self.layer3(x)
        else:
            raise Exception
        return x


class prediction_MLP(nn.Module):
    def __init__(self, in_dim=2048, hidden_dim=512, out_dim=2048): # bottleneck structure
        super().__init__()
        ''' page 3 baseline setting
        Prediction MLP. The prediction MLP (h) has BN applied
        to its hidden fc layers. Its output fc does not have BN
        (ablation in Sec. 4.4) or ReLU. This MLP has 2 layers.
        The dimension of h’s input and output (z and p) is d = 2048,
        and h’s hidden layer’s dimension is 512, making h a
        bottleneck structure (ablation in supplement).
        '''
        self.layer1 = nn.Sequential(
            nn.Linear(in_dim, hidden_dim),
            nn.BatchNorm1d(hidden_dim),
            nn.ReLU(inplace=True)
        )
        self.layer2 = nn.Linear(hidden_dim, out_dim)
        """
        Adding BN to the output of the prediction MLP h does not work
        well (Table 3d). We find that this is not about collapsing.
        The training is unstable and the loss oscillates.
        """

    def forward(self, x):
        x = self.layer1(x)
        x = self.layer2(x)
        return x

class SimSiam(nn.Module):
    def __init__(self, config, backbone='resnet50', _pnn=None):
        super().__init__()
        if backbone == 'swin':
            self.backbone = swin(
                img_size=config.dataset.image_size,#config.DATA.IMG_SIZE,
                patch_size=config.model.swin.patch_size,#config.MODEL.SWIN.PATCH_SIZE,
                in_chans=config.model.swin.in_chans, #MODEL.SWIN.IN_CHANS,
                num_classes=0,
                embed_dim=config.model.swin.embed_dim,
                depths=config.model.swin.depths,
                num_heads=config.model.swin.num_heads,
                window_size=config.model.swin.window_size,
                mlp_ratio=config.model.swin.mlp_ratio,
                qkv_bias=config.model.swin.qkv_bias,
                qk_scale=config.model.swin.qk_scale,
                drop_rate=config.model.swin.drop_rate,
                drop_path_rate=config.model.swin.drop_path_rate,
                ape=config.model.swin.ape,
                patch_norm=config.model.swin.patch_norm,
                use_checkpoint=config.train.use_checkpoint)       
            # import pdb; pdb.set_trace()     
            self.projector = projection_MLP(768)
        elif backbone == 'vit':
            self.backbone = vit(
                img_size=config.dataset.image_size,
                patch_size=config.model.vit.patch_size,
                in_chans=config.model.vit.in_chans,
                num_classes=0,
                embed_dim=config.model.vit.embed_dim,
                depth=config.model.vit.depth,
                num_heads=config.model.vit.num_heads,
                mlp_ratio=config.model.vit.mlp_ratio,
                qkv_bias=config.model.vit.qkv_bias,
                drop_rate=config.model.vit.drop_rate,
                drop_path_rate=config.model.vit.drop_path_rate,
                norm_layer=partial(nn.LayerNorm, eps=1e-6),
                init_values=config.model.vit.init_values,
                use_abs_pos_emb=config.model.vit.use_ape,
                use_rel_pos_bias=config.model.vit.use_rpb,
                use_shared_rel_pos_bias=config.model.vit.use_shared_rpb,
                use_mean_pooling=config.model.vit.use_mean_pooling)   
            # import pdb; pdb.set_trace()     
            self.projector = projection_MLP(768)
        else:
            self.backbone = eval(f"{backbone}")()        
            self.projector = projection_MLP(self.backbone.fc.in_features)
        self.encoder = nn.Sequential( # f encoder
            self.backbone,
            self.projector
        )
        self.predictor = prediction_MLP()

    def forward(self, x1, x2, return_emb=False):
        f, h = self.encoder, self.predictor     
        b1 = self.backbone(x1)
        z1 = self.projector(b1)
        
        z2 = f(x2)
        p1, p2 = h(z1), h(z2)
        
        L = D(p1, z2) / 2 + D(p2, z1) / 2
        # return {'loss': L}
        if return_emb:
            return L, b1
        else:
            return L

def build_simsiam(config):    
    model = SimSiam(config=config, backbone=config.model.type)
    return model
