#!/bin/bash
HOMEPAGE_URL_FILE="./config/webarena_urls/homepage_url.txt" # app.py automatically creates this file
# export PYTHONPATH=$(pwd)


#===============================================================================
# AUXILIARY FUNCTIONS
#===============================================================================
build_url_from_docker() {
  local container_name=$1
  local base_url=$2

  if docker ps --filter "name=$container_name" --format '{{.Names}}' | grep -w $container_name &> /dev/null; then
    # Extract port mappings from the HostConfig.PortBindings section of the Docker inspect output.
    local port_bindings=$(docker inspect --format='{{range $p, $conf := .HostConfig.PortBindings}}{{$p}}:{{(index $conf 0).HostPort}}{{"\n"}}{{end}}' "$container_name")
    # Get the first port mapping
    local port=$(echo "$port_bindings" | head -n 1 | cut -d ':' -f2)

    # If the port is not set or is null, website is not available
    if [ -z "$port" ] || [ "$port" = "null" ]; then
      echo "PASS"
    else
    # Obs: do not add a slash `/` on the end of paths, otherwise autologin code will fail.
      echo "http://$base_url:$port"
    fi
  
  # If the container is not running, website is not available
  else
    echo "PASS"
  fi
}

get_homepage_url(){
    # read txt file in  ./config/webarena_urls/homepage.txt
    if [ -f "$HOMEPAGE_URL_FILE" ]; then
        local homepage=$(cat "$HOMEPAGE_URL_FILE")
        # Check if the homepage is reachable
        if curl --output /dev/null --silent --head --fail "$homepage"; then
            echo "$homepage"
        else
            echo "PASS"
        fi
    else
        echo "PASS"
    fi
}
# Function to print the environment variables
print_vars() {
    local benchmark=$1
    echo -e "\n==================================\nEnvironment variables set\n=================================="

    if [ "$benchmark" = "local_webarena" ]; then
        printf "%-50s %s\n" "SHOPPING: $SHOPPING" "(container name: 'shopping')"
        printf "%-50s %s\n" "REDDIT: $REDDIT" "(container name: 'forum')"
        printf "%-50s %s\n" "WIKIPEDIA: $WIKIPEDIA" "(container name: 'wikipedia')"
        printf "%-50s %s\n" "GITLAB: $GITLAB" "(container name: 'gitlab')"
        printf "%-50s %s\n" "MAP: $MAP" ""
        printf "%-50s %s\n" "SHOPPING_ADMIN: $SHOPPING_ADMIN" "(container name: 'shopping_admin')"
        printf "%-50s %s\n" "HOMEPAGE: $HOMEPAGE" "(FLASK app. URL in  ./config/webarena_urls/homepage.txt)"
    elif [ "$benchmark" = "local_vwebarena" ]; then
        printf "%-50s %s\n" "SHOPPING: $SHOPPING" "(container name: 'shopping')"
        printf "%-50s %s\n" "REDDIT: $REDDIT" "(container name: 'forum')"
        printf "%-50s %s\n" "CLASSIFIEDS: $CLASSIFIEDS" "(container name: 'classifieds')"
        printf "%-50s %s\n" "WIKIPEDIA: $WIKIPEDIA" "(container name: 'wikipedia')"
        printf "%-50s %s\n" "CLASSIFIEDS_RESET_TOKEN: $CLASSIFIEDS_RESET_TOKEN" ""
        printf "%-50s %s\n" "HOMEPAGE: $HOMEPAGE" "(FLASK app. URL in  ./config/webarena_urls/homepage.txt)"
    
    elif [ "$benchmark" = "demo_webarena" ]; then
        printf "%-50s %s\n" "SHOPPING: $SHOPPING" ""
        printf "%-50s %s\n" "REDDIT: $REDDIT" ""
        printf "%-50s %s\n" "WIKIPEDIA: $WIKIPEDIA" ""
        printf "%-50s %s\n" "GITLAB: $GITLAB" ""
        printf "%-50s %s\n" "MAP: $MAP" ""
        printf "%-50s %s\n" "SHOPPING_ADMIN: $SHOPPING_ADMIN" ""
        printf "%-50s %s\n" "HOMEPAGE: $HOMEPAGE" ""
    
    elif [ "$benchmark" = "demo_vwebarena" ]; then
        printf "%-50s %s\n" "SHOPPING: $SHOPPING" ""
        printf "%-50s %s\n" "REDDIT: $REDDIT" ""
        printf "%-50s %s\n" "CLASSIFIEDS: $CLASSIFIEDS" ""
        printf "%-50s %s\n" "WIKIPEDIA: $WIKIPEDIA" ""
        printf "%-50s %s\n" "CLASSIFIEDS_RESET_TOKEN: $CLASSIFIEDS_RESET_TOKEN" ""
        printf "%-50s %s\n" "HOMEPAGE: $HOMEPAGE" ""
    else
        echo "Benchmark '$benchmark' not recognized."
    
    fi
}


# Function to export environment variabless
export_vars() {
  local mode=$1
  local base_url=$2

  if [ -z "$base_url" ]; then
    base_url="localhost"
  fi

  # Websites hosted by WebArena authors
  if [ "$mode" = "demo_webarena" ]; then
    # Obs: do not add a slash `/` on the end of paths, otherwise autologin code will fail.
    export REDDIT="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:9999"
    export CLASSIFIEDS="PASS"
    export CLASSIFIEDS_RESET_TOKEN="4b61655535e7ed388f0d40a93600254c"
    export SHOPPING="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:7770" # NOTE: not the same as vwebarena!! Products are different
    export GITLAB="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:8023"
    export MAP="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:3000"
    export WIKIPEDIA="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:8888"
    export SHOPPING_ADMIN="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:7780/admin"
    export HOMEPAGE="PASS"

  # Websites hosted by VWebArena authors
  elif [ "$mode" = "demo_vwebarena" ]; then
   # Obs: do not add a slash `/` on the end of paths, otherwise autologin code will fail.
    export CLASSIFIEDS="http://ec2-3-13-232-171.us-east-2.compute.amazonaws.com:9980"
    export CLASSIFIEDS_RESET_TOKEN="4b61655535e7ed388f0d40a93600254c"
    export SHOPPING="http://localhost:7770"  # NOTE: not the same as webarena!! Products are different
    export REDDIT="http://ec2-3-13-232-171.us-east-2.compute.amazonaws.com:9999"
    export WIKIPEDIA="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:8888"
    export GITLAB="PASS"
    export MAP="PASS"
    export SHOPPING_ADMIN="PASS"
    export HOMEPAGE="PASS"

  # Get URL from environments running locally - VWebArena
  elif [ "$mode" = "local_vwebarena" ]; then
    export CLASSIFIEDS=$(build_url_from_docker classifieds $base_url)
    export CLASSIFIEDS_RESET_TOKEN="4b61655535e7ed388f0d40a93600254c"
    export SHOPPING=$(build_url_from_docker shopping $base_url)
    export REDDIT=$(build_url_from_docker forum $base_url)
    export GITLAB="PASS"
    export MAP="PASS"
    export SHOPPING_ADMIN="PASS"
    export HOMEPAGE=$(get_homepage_url)

    wiki_url=$(build_url_from_docker wikipedia $base_url)
    if [ "$wiki_url" = "PASS" ]; then
        export WIKIPEDIA="PASS"
    else
        export WIKIPEDIA=$wiki_url
    fi
    echo -e "\nNOTE:Websites set to PASS will not be considered for evaluation. \
If any website is unexpectedly marked as PASS, check if the corresponding containers \
are running or export the environment variables manually before running the tests."
  
  # Get URL from environments running locally - WebArena
  elif [ "$mode" = "local_webarena" ]; then
    export CLASSIFIEDS="PASS"
    export CLASSIFIEDS_RESET_TOKEN="4b61655535e7ed388f0d40a93600254c"
    export SHOPPING=$(build_url_from_docker shopping $base_url)
    export REDDIT=$(build_url_from_docker forum $base_url)
    export GITLAB=$(build_url_from_docker gitlab $base_url)
    export MAP="http://ec2-3-131-244-37.us-east-2.compute.amazonaws.com:3000" # Hosted by the authors; see environments_docker Readme
    export SHOPPING_ADMIN=$(get_url shopping_admin $base_url)
    export HOMEPAGE=$(get_homepage_url)

    wiki_url=$(build_url_from_docker wikipedia $base_url)
    if [ "$wiki_url" = "PASS" ]; then
        export WIKIPEDIA="PASS"
    else
        export WIKIPEDIA=$wiki_url
    fi
    echo -e "\nNOTE:Websites set to PASS will not be considered for evaluation. \
If any website is unexpectedly marked as PASS, check if the corresponding containers \
are running or export the environment variables manually before running the tests."

  else
    echo "Mode '$mode' not recognized."
    exit 1
  
  fi

  print_vars $mode
}

#===============================================================================
# Usage: ./set_env_variables.sh <mode> [base_url]
#===============================================================================
if [ $# -lt 1 ] || [ $# -gt 2 ]; then
  echo "Usage: $0 <mode> [base_url]"
  exit 1
fi
env_name=$1
base_url=$2
export_vars $env_name $base_url
