import os
import requests
import time
import numpy as np
EPS = np.finfo(np.float32).eps
from typing import Dict

from eai_distributed_toolkit.toolkit_utils import add_url_prefix

class MinMaxLayer():

  def __init__(self, vmin, vmax):
    self._min = vmin.astype(np.float32)
    self._max = vmax.astype(np.float32)
    self._mean_range = (self._min + self._max) / 2.0
    self._half_range = (0.5*(self._max - self._min))
    # Half_range shouldn't already be negative.
    self._half_range = np.clip(self._half_range, EPS, 1e7)


class MinMaxNormalizationLayer(MinMaxLayer):
  """Maps an un-normalized vector to -1, 1."""

  def __call__(self, vector, **kwargs):
    return (vector - self._mean_range) / self._half_range


class MinMaxDenormalizationLayer(MinMaxLayer):
  """Maps -1, 1 vector back to un-normalized."""

  def __call__(self, vector, **kwargs):

    return (vector * self._half_range) + self._mean_range


def normalise_data_min_max(data, min_data, max_data):
    return 2.0 * (data - min_data) / (max_data - min_data) - 1


def normalise_data_mean_stddev(data, mean_data, stddev_data):
    return (data - mean_data) / np.clip(stddev_data, EPS, 1e7)


def avg_rtg(rtg, dones, max_ep_len):
    avg_rtg = np.inf * np.ones_like(rtg)
    step_count = 0
    for i, (cum_rew, done) in enumerate(zip(rtg, dones)):
        avg_rtg[i] = cum_rew / (max_ep_len - step_count)
        step_count += 1
        if done:
            step_count = 0

    return avg_rtg


def get_job_config() -> Dict:
    toolkit_url = "/v1/me"
    console_url = os.environ.get("EAI_CONSOLE_URL")
    if console_url:
        url = f"{add_url_prefix(console_url, 'internal')}{toolkit_url}"
    else:
        raise EnvironmentError("What is borgy??")

    config = {}
    for _ in range(60):
        r = requests.get(url)
        if r.status_code == 200:
            config = r.json()
            break
        time.sleep(1)
    if len(config) == 0:
        print(f"Can't find Ork job config! {r} on {toolkit_url}: {r.text}, {config}")
    return config
