#ifndef LOGNUM_HPP
#define LOGNUM_HPP

#include <iostream>
#include <iomanip>
#include <cmath>

using namespace std;

typedef long double float_type;

float_type log_sum_exp(float_type a, float_type b) {
	if (!isfinite(a)) return b;
	if (!isfinite(b)) return a;
	float_type x = max(a, b);
	return x + log(exp(a - x) + exp(b - x));
}

void log_sum_exp_r(float_type& a, float_type b) {
	if (!isfinite(a)) {
        a = b;
        return;
    }
	if (!isfinite(b)) return;
	float_type x = max(a, b);
	a = x + log(exp(a - x) + exp(b - x));
}

float_type log_sub_exp(float_type a, float_type b) {
	if (!isfinite(a)) return b;
	if (!isfinite(b)) return a;
	float_type x = max(a, b);
	return x + log(exp(a - x) - exp(b - x));
}

void log_sub_exp_r(float_type& a, float_type b) {
	if (!isfinite(a)) {
        a = b;
        return;
    }
	if (!isfinite(b)) return;
	float_type x = max(a, b);
	a = x + log(exp(a - x) - exp(b - x));
}

// Class LogNum

class LogNum { 
    public:
    float_type value;
	bool positive;

	LogNum& operator=(const float_type z) {
		value = log(abs(z));
		positive = z >= 0;
		return *this;
	}

	LogNum& operator=(LogNum z) {
		value = z.value;
		positive = z.positive;
		return *this;
	}

	inline LogNum operator-() {
        LogNum z;
        z.value = value;
		z.positive = !positive;
        return z;
    }

	inline LogNum& operator+=(const LogNum y){
		float_type new_value;
		bool new_positive;
		if (positive == y.positive) {
			new_value = log_sum_exp(value, y.value);
			new_positive = positive;
		} else {
			new_value = log_sub_exp(max(value, y.value), min(value, y.value));
			new_positive = (y.value < value ? positive : y.positive);
		}
		positive = new_positive;
		value = new_value;
		return *this;
	}

	inline LogNum& operator-=(const LogNum y){
		float_type new_value;
		bool new_positive;
		if (positive == !y.positive) {
			new_value = log_sum_exp(value, y.value);
			new_positive = positive;
		} else {
			new_value = log_sub_exp(max(value, y.value), min(value, y.value));
			new_positive = (y.value < value ? positive : !y.positive);
		}
		positive = new_positive;
		value = new_value;
		return *this;
	}

	inline LogNum& operator*=(const LogNum y){
		value += y.value;
		positive = !(positive ^ y.positive);
		return *this;
	}

	inline LogNum operator*=(const float_type x){
		value += log(abs(x));
		positive = !(positive ^ (x > 0));
		return *this;
	}

	void set_log(const float_type x) {
		value = x;
	}
};

inline LogNum operator+(LogNum x, LogNum y){
	x += y;
	return x;
}

inline LogNum operator-(LogNum x, LogNum y){
	x -= y;
	return x;
}

inline LogNum operator*(LogNum x, LogNum y){
	return { x.value + y.value, !(x.positive ^ y.positive) };	
}

inline LogNum operator*(float_type x, LogNum y){
	LogNum z;
	z = x;
	return z * y;
}

inline LogNum operator/(LogNum x, LogNum y){
	return { x.value - y.value, !(x.positive ^ y.positive) };	
}

inline bool operator<(LogNum x, LogNum y){
	if (!x.positive && y.positive) return true;
	if (x.positive && !y.positive) return false;
	if (x.positive) return x.value < y.value;
	return x.value > y.value;
}

inline bool operator<(LogNum x, float_type y){
	if (!x.positive && y > 0) return true;
	if (x.positive && y < 0) return false;
	if (x.positive) return x.value < log(y);
	return x.value > log(-y);
}

ostream& operator<<(ostream& os, LogNum x){
	if (x.positive) os<<x.value;
    else os<<"~"<<x.value;
	return os;
}

#endif