import os
import abc
import time
import numpy.random
import torch
import pickle
import numpy as np
import gymnasium as gym
from gymnasium import utils
from gymnasium.envs.mujoco import mujoco_env
from gymnasium.envs.mujoco.inverted_pendulum import InvertedPendulumEnv
from imitations.data.types import TrajectoryWithRew

# =========================================================================== #
#                         Inverted Pendulum Wall                              #
# =========================================================================== #


class InvertedPendulumWall(InvertedPendulumEnv):

    def step(self, a):
        xposbefore = self.sim.data.qpos[0]
        self.do_simulation(a, self.frame_skip)
        xposafter = self.sim.data.qpos[0]
        obs = self._get_obs()
        notdone = np.isfinite(obs).all() and (np.abs(obs[1]) <= 0.2)
        done = not notdone
        terminated = done
        truncated = done
        info = {'x_position': xposafter}
        if xposafter <= -0.01:
            reward = 0.1
        elif xposafter >= 0:
            reward = 1
        else:
            reward = (-xposafter/0.01)*0.9+0.1
        return obs, reward, terminated, truncated, info

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(
            size=self.model.nq, low=-0.01, high=0.01
        )
        qvel = self.init_qvel + self.np_random.uniform(
            size=self.model.nv, low=-0.01, high=0.01
        )
        self.set_state(qpos, qvel)
        return self._get_obs()
    
    def rollout(self):

        folder_path = r'.\expert_data\BiasedPendulumn'
        all_trajectories = []

        for file_name in os.listdir(folder_path):
            if file_name.endswith('.pkl'):
                file_path = os.path.join(folder_path, file_name)
                with open(file_path, 'rb') as file:
                    data = pickle.load(file)

                obs = data['observations']
                # obs = obs[:, -8:]
                acts = data['actions'][:len(data['actions'])-1]
                rews = data['reward'][:len(data['actions'])-1]
                terminal = True
                infos = None

                trajectory = TrajectoryWithRew(obs=obs, acts=acts, rews=rews, terminal=terminal, infos=infos)
                all_trajectories.append(trajectory)

        return all_trajectories
    

class InvertedPendulumWallNoise(InvertedPendulumWall):
    def __init__(self, noise_mean, noise_std, noise_seed: int = 0):
        self.noise_mean = noise_mean
        self.noise_std = noise_std
        self.noise_seed = noise_seed
        self.rdm = np.random.RandomState(1)
        super().__init__()

    def step(self, a):
        xposbefore = self.sim.data.qpos[0]
        self.do_simulation(a, self.frame_skip)

        noise_qpos = self.rdm.uniform(-(self.noise_mean+self.noise_std), self.noise_mean+self.noise_std, self.model.nq)
        noise_qvel = self.rdm.uniform(-(self.noise_mean+self.noise_std), self.noise_mean+self.noise_std, self.model.nv)

        qpos = self.sim.data.qpos + noise_qpos
        qvel = self.sim.data.qvel + noise_qvel

        self.set_state(qpos=qpos, qvel=qvel)

        xposafter = self.sim.data.qpos[0]
        obs = self._get_obs()
        notdone = np.isfinite(obs).all() and (np.abs(obs[1]) <= 0.2)
        done = not notdone
        info = {'x_position': xposafter}
        if xposafter <= -0.01:
            reward = 1
        elif xposafter >= 0:
            reward = 0.1
        else:
            reward = (-xposafter/0.01)*0.9+0.1
        return obs, reward, done, info

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(
            size=self.model.nq, low=-0.01, high=0.01
        )
        qvel = self.init_qvel + self.np_random.uniform(
            size=self.model.nv, low=-0.01, high=0.01
        )
        self.set_state(qpos, qvel)
        return self._get_obs()
