import torch
import torch.nn as nn
import torch.nn.functional as F

class Swish(nn.Module): 
    def __init__(self):
        super().__init__()

    def forward(self, input):
        return input * torch.sigmoid(input)


class IdentityLayer(nn.Module):
    def forward(self, x):
        return x
       
class ResNetBlock(nn.Module):
    def __init__(self, in_size=16, out_size=16, downsample = False):
        super(ResNetBlock,self).__init__()
        self.out_size = out_size
        self.in_size = in_size
        if downsample:
            self.stride1 = 2
            self.reslayer = nn.Conv2d(in_channels=self.in_size, out_channels=self.out_size, stride=2, kernel_size=1)
        else:
            self.stride1 = 1
            self.reslayer = IdentityLayer()
        self.conv1 =nn.Conv2d(in_channels=self.in_size,out_channels=self.out_size,kernel_size=3,stride=self.stride1,padding=1)
        self.conv2 =nn.Conv2d(in_channels=self.out_size,out_channels=self.out_size,kernel_size=3,padding=1)
        self.bn1 = nn.BatchNorm2d(out_size)
        self.bn2 = nn.BatchNorm2d(out_size)

    def forward(self, x):
        residual = self.reslayer(x)
        out = F.relu(self.bn1(self.conv1(x)), inplace=True)
        out = self.bn2(self.conv2(out))
        out += residual
        out = F.relu(out)
        return out


class ConvNet(nn.Module):
    def __init__(self, in_channels, out_dim, im_size = (32,32)):
        super(ConvNet, self).__init__()
        net_width, net_depth, net_act, net_norm, net_pooling = 128, 3, 'relu', 'instancenorm', 'avgpooling'
        self.features, shape_feat = self._make_layers(in_channels, net_width, net_depth, net_norm, net_act, net_pooling, im_size)
        num_feat = shape_feat[0]*shape_feat[1]*shape_feat[2]
        self.classifier = nn.Linear(num_feat, out_dim)

    def forward(self, x):
        out = self.features(x)
        feature = out.view(out.size(0), -1)
        out = self.classifier(feature)
        return feature, out

    def get_features(self, x):
        out = self.features(x)
        out = out.view(out.size(0), -1)
        return out

    def _get_activation(self, net_act):
        if net_act == 'sigmoid':
            return nn.Sigmoid()
        elif net_act == 'relu':
            return nn.ReLU(inplace=True)
        elif net_act == 'leakyrelu':
            return nn.LeakyReLU(negative_slope=0.01)
        elif net_act == 'swish':
            return Swish()
        else:
            exit('unknown activation function: %s'%net_act)

    def _get_pooling(self, net_pooling):
        if net_pooling == 'maxpooling':
            return nn.MaxPool2d(kernel_size=2, stride=2)
        elif net_pooling == 'avgpooling':
            return nn.AvgPool2d(kernel_size=2, stride=2)
        elif net_pooling == 'none':
            return None
        else:
            exit('unknown net_pooling: %s'%net_pooling)

    def _get_normlayer(self, net_norm, shape_feat):
        if net_norm == 'batchnorm':
            return nn.BatchNorm2d(shape_feat[0], affine=True)
        elif net_norm == 'layernorm':
            return nn.LayerNorm(shape_feat, elementwise_affine=True)
        elif net_norm == 'instancenorm':
            return nn.GroupNorm(shape_feat[0], shape_feat[0], affine=True)
        elif net_norm == 'groupnorm':
            return nn.GroupNorm(4, shape_feat[0], affine=True)
        elif net_norm == 'none':
            return None
        else:
            exit('unknown net_norm: %s'%net_norm)

    def _make_layers(self, channel, net_width, net_depth, net_norm, net_act, net_pooling, im_size):
        layers = []
        in_channels = channel
        if im_size[0] == 28:
            im_size = (32, 32)
        shape_feat = [in_channels, im_size[0], im_size[1]]
        for d in range(net_depth):
            layers += [nn.Conv2d(in_channels, net_width, kernel_size=3, padding=3 if channel == 1 and d == 0 else 1)]
            shape_feat[0] = net_width
            if net_norm != 'none':
                layers += [self._get_normlayer(net_norm, shape_feat)]
            layers += [self._get_activation(net_act)]
            in_channels = net_width
            if net_pooling != 'none':
                layers += [self._get_pooling(net_pooling)]
                shape_feat[1] //= 2
                shape_feat[2] //= 2

        return nn.Sequential(*layers), shape_feat


class LeNet(nn.Module):
    def __init__(self, in_channels, out_dim, out_channels=16):
        super(LeNet, self).__init__()
        self.conv_layers = nn.Sequential(
            nn.Conv2d(in_channels, out_channels, 3, padding=1),
            nn.ReLU(),
            nn.MaxPool2d(2, 2),
            nn.Conv2d(out_channels, 2*out_channels, 3, padding=1),
            nn.ReLU(),
            nn.MaxPool2d(2, 2)
        )
        self.fc_layers = nn.Sequential(
            nn.Flatten(),
            nn.Linear(2*out_channels*8*8, 108),
            nn.ReLU(),
            nn.Linear(108, 2048),
            nn.ReLU()
        )
        self.fc = nn.Linear(2048, out_dim)

    def get_features(self, x):
        features = self.conv_layers(x)
        return features.flatten(1)  

    def forward(self, x):
        x = self.conv_layers(x)
        feature = x.view(x.size(0), -1)
        out = self.fc_layers(feature)
        out = self.fc(out)
        return feature, out


class VGG8(nn.Module):
    def __init__(self, in_channels, out_dim, out_channels=16):
        super(VGG8, self).__init__()
        self.features = nn.Sequential(
            nn.Conv2d(in_channels, out_channels, kernel_size=3, padding=1),
            nn.ReLU(inplace=True),
            nn.Conv2d(out_channels, out_channels, kernel_size=3, padding=1),
            nn.ReLU(inplace=True),
            nn.MaxPool2d(kernel_size=2, stride=2),
            nn.Conv2d(out_channels, out_channels*2, kernel_size=3, padding=1),
            nn.ReLU(inplace=True),
            nn.Conv2d(out_channels*2, out_channels*2, kernel_size=3, padding=1),
            nn.ReLU(inplace=True),
            nn.MaxPool2d(kernel_size=2, stride=2),
            nn.Conv2d(out_channels*2, out_channels*4, kernel_size=3, padding=1),
            nn.ReLU(inplace=True),
            nn.Conv2d(out_channels*4, out_channels*4, kernel_size=3, padding=1),
            nn.ReLU(inplace=True),
            nn.MaxPool2d(kernel_size=2, stride=2)
        )
        self.classifier = nn.Sequential(
            nn.Linear(out_channels*4 * 4 * 4, 180),
            nn.ReLU(inplace=True),
            nn.Dropout(),
            nn.Linear(180, 2048),
            nn.ReLU(inplace=True),
            nn.Dropout(),
            nn.Linear(2048, out_dim)
        )
    
    def get_features(self, x):
        out = self.features(x)
        feature = out.view(out.size(0), -1)
        return feature

    def forward(self, x):
        x = self.features(x)
        feature = x.view(x.size(0), -1)
        out = self.classifier(feature)
        return feature, out
    

class MLP(nn.Module):
    def __init__(self, in_channels, out_dim):
        super(MLP, self).__init__()
        self.fc1 = nn.Linear( in_channels * 32 * 32, 128)
        self.fc2 = nn.Linear(128, 2048)
        self.fc3 = nn.Linear(2048, out_dim)

    def get_features(self, x):
        x = x.view(x.shape[0], -1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        return x

    def forward(self, x):
        x = x.view(x.shape[0], -1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        feature = x.view(x.size(0), -1)
        out = self.fc3(feature)
        return feature, out
    

class ResNet9(nn.Module):
    def __init__(self,in_channels,out_dim):
        super(ResNet9, self).__init__()
        self.conv1 = nn.Conv2d(in_channels, 32, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(32)
        
        self.filter_size = [[32,32], [32,32], [32,32], [32,64], 
                    [64,64], [64,64], [64,128], [128,128], [128,128]]
        layers = []
        for i in range(9):
            down_sample = False
            if i == 3 or i == 6 :
                down_sample = True
            layers.append(ResNetBlock(self.filter_size[i][0], self.filter_size[i][1],downsample=down_sample))
        self.res_net = nn.Sequential(*layers)
        self.global_avg = nn.AvgPool2d(2)
        self.fc3 = nn.Linear(2048,out_dim)
    
    def get_features(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = self.res_net(x)
        x = self.global_avg(x)
        feature = x.view(x.shape[0], -1)
        return feature

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = self.res_net(x)
        x = self.global_avg(x)
        feature= x.view(x.shape[0], -1)
        out = self.fc3(feature)
        return feature, out
    
