function [timeInt,timePoint,res] = reach_fromStart(obj,params,options)
% reach_fromStart - computes the reachable set for linear systems using the
%    propagation of the homogeneous solution from the start
%
% Syntax:
%    [timeInt,timePoint,res] = reach_fromStart(obj,params,options)
%
% Inputs:
%    obj - linearSys object
%    params - model parameters
%    options - options for the computation of reachable sets
%
% Outputs:
%    timeInt - array of time-interval reachable / output sets
%    timePoint - array of time-point reachable / output sets
%    res - true/false whether specification satisfied
%
% Example:
%    -
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: none

% Authors:       Mark Wetzlinger
% Written:       26-June-2019
% Last update:   14-August-2019
%                16-February-2021 (MW, correct implementation of uTransVec)
%                19-November-2022 (MW, modularize specification check)
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

% obtain factors for initial state and input solution
for i=1:(options.taylorTerms+1)
    % compute initial state factor
    options.factor(i) = options.timeStep^(i)/factorial(i);
end

% if a trajectory should be tracked
if isfield(params,'uTransVec')
    params.uTrans = params.uTransVec(:,1);
end

% log information
verboseLog(options.verbose,1,params.tStart,params.tStart,params.tFinal);

% init step 
[Rnext, options] = initReach_Euclidean(obj, params.R0, params, options);
% init for loop
Rtrans = options.Rtrans;
Rinhom = options.Rpar;
Raux = options.Raux;
if ~isfield(params,'uTransVec')
    Rhom_0 = options.Rhom;
    Rhom_tp_0 = options.Rhom_tp;
    inputCorr = 0;
else
    Rhom_tp = options.Rhom_tp;
end
eADelta = obj.taylor.eAt;
P = eye(obj.dim);
Q = obj.taylor.eAt;


% time period, number of steps, step counter
tVec = params.tStart:options.timeStep:params.tFinal;
steps = length(tVec) - 1;
options.i = 1;

% initialize output variables for reachable sets and output sets
timeInt.set = cell(steps,1);
timeInt.time = cell(steps,1);
timePoint.set = cell(steps+1,1);
timePoint.time = num2cell(tVec');

% compute output set of first step
timePoint.set{1} = outputSet(obj,params.R0,params,options);
timeInt.set{1} = outputSet(obj,Rnext.ti,params,options);
timeInt.time{1} = interval(tVec(1),tVec(2));
Rstart = Rnext.tp;

% safety property check
if isfield(options,'specification')
    [res,timeInt,timePoint] = checkSpecification(...
        options.specification,Rnext.ti,timeInt,timePoint,1);
    if ~res; return; end
end


for i = 2:steps
    
    options.i = i;
    
    % post --------------
    
    % if a trajectory should be tracked
    if isfield(params,'uTransVec')
        params.uTrans = params.uTransVec(:,i);
        options.Rhom_tp = Rhom_tp;
        % recompute Rtrans/inputCorr -> also propagate Rhom/Rhom_tp
        [Rhom,Rhom_tp,Rtrans,inputCorr] = inputInducedUpdates(obj,params,options);
        % reduce homogeneous part
        Rhom_tp = reduce(Rhom_tp,options.reductionTechnique,options.zonotopeOrder);
        % propagate inhomogeneous part
        Rinhom = eADelta * Rinhom + Raux;
    else
        % propagate homogeneous part (from start -> take Rhom_0)
        Rhom    = Q * Rhom_0;
        Rhom_tp = Q * Rhom_tp_0;
        % no uTransVec -> Rtrans is constant
        Rinhom = Rinhom + Q * Raux + P * Rtrans;
    end
    
    % reduce (only inhomogeneous solution)
    Rinhom = reduce(Rinhom,options.reductionTechnique,options.zonotopeOrder);
    
    % R([t_k, t_k + Delta t_k]) = H([t_k, t_k + Delta t_k]) + P([0, t_k])
    Rnext.ti = Rhom + Rinhom + inputCorr;
    Rnext.tp = Rhom_tp + Rinhom;
    
    % propagate matrix exponentials
    P = Q;
    Q = Q * eADelta;
    
    % --------------

    % compute output set
    timePoint.set{i} = outputSet(obj,Rstart,params,options);
    timeInt.set{i} = outputSet(obj,Rnext.ti,params,options);
    timeInt.time{i} = interval(tVec(i),tVec(i+1));
    
    % save reachable set
    Rstart = Rnext.tp;
    
    % safety property check
    if isfield(options,'specification')
        [res,timeInt,timePoint] = checkSpecification(...
            options.specification,Rnext.ti,timeInt,timePoint,i);
        if ~res; return; end
    end
    
    % log information
    verboseLog(options.verbose,i,tVec(i),params.tStart,params.tFinal);
    
end

% compute output set of last set
timePoint.set{end} = outputSet(obj,Rstart,params,options);

% specification fulfilled
res = true;

% ------------------------------ END OF CODE ------------------------------
