function [timeInt,timePoint,res] = reach_wrappingfree(sys,params,options)
% reach_wrappingfree - computes the reachable set for linear systems using
%    the wrapping-free reachability algorithm for linear systems [1]
%
% Syntax:
%    [timeInt,timePoint,res] = reach_wrappingfree(sys,params,options)
%
% Inputs:
%    sys - continuous system object
%    params - model parameters
%    options - options for the computation of reachable sets
%
% Outputs:
%    timeInt - array of time-interval reachable / output sets
%    timePoint - array of time-point reachable / output sets
%    res - true/false whether specification satisfied
%
% References:
%    [1] A. Girard, C. Le Guernic, and O. Maler, "Efficient computation of
%        reachable sets of linear time-invariant systems with inputs"
%        in Hybrid Systems: Computation and Control, ser. LNCS 3927.
%        Springer, 2006, pp. 257--271.
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: none

% Authors:       Matthias Althoff, Mark Wetzlinger
% Written:       26-June-2019 (from @contDynamics > reach.m)
% Last update:   14-August-2019
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

%obtain factors for initial state and input solution
for i=1:(options.taylorTerms+1)
    options.factor(i) = options.timeStep^(i)/factorial(i);
end

% if a trajectory should be tracked
tracking = isfield(params,'uTransVec');
if tracking
    params.uTrans = params.uTransVec(:,1);
else
    inputCorr = 0;
end

% log information
verboseLog(options.verbose,1,params.tStart,params.tStart,params.tFinal);

% initialize reachable set computations
[Rnext, options] = initReach_Euclidean(sys, params.R0, params, options);
Rtrans = options.Rtrans;
Rhom   = options.Rhom;
Rhom_tp = options.Rhom_tp;
Rpar   = options.Rpar;
Raux   = options.Raux;
eAt    = sys.taylor.eAt;

% time period and number of steps
tVec = params.tStart:options.timeStep:params.tFinal;
steps = length(tVec) - 1;
options.i = 1;

% initialize output variables for reachable sets and output sets
timeInt.set = cell(steps,1);
timeInt.time = cell(steps,1);
timePoint.set = cell(steps+1,1);
timePoint.time = num2cell(tVec');

% compute output set of first step
timePoint.set{1} = outputSet(sys,params.R0,params,options);
timeInt.set{1} = outputSet(sys,Rnext.ti,params,options);
timeInt.time{1} = interval(tVec(1),tVec(2));
Rstart = Rnext.tp;

% safety property check
if isfield(options,'specification')
    [res,timeInt,timePoint] = checkSpecification(...
        options.specification,Rnext.ti,timeInt,timePoint,1);
    if ~res; return; end
end


% loop over all reachability steps
for i = 2:steps
    
    options.i = i;
    
    % post: (wrapping free) ----------
    
    % method implemented from Algorithm 2 in [1]
    
    % if a trajectory should be tracked
    if tracking
        params.uTrans = params.uTransVec(:,i);
        options.Rhom_tp = Rhom_tp;
        [Rhom,Rhom_tp,Rtrans,inputCorr] = inputInducedUpdates(sys,params,options);
    else
        Rhom = eAt*Rhom + center(Rtrans);
        Rhom_tp = eAt*Rhom_tp + center(Rtrans);
    end
    Raux = eAt*Raux;
    Rpar = Rpar + interval(Raux) + interval(Rtrans) + (-center(Rtrans));
    
    %write results to reachable set struct Rnext
    if isa(Rhom,'polytope')
        Rnext.ti = Rhom + polytope(Rpar) + polytope(inputCorr);
        Rnext.tp = Rhom_tp + polytope(Rpar);
    else
        Rnext.ti = Rhom + zonotope(Rpar) + inputCorr;
        Rnext.tp = Rhom_tp + zonotope(Rpar);
    end
    
    % ----------

    % compute output set
    timePoint.set{i} = outputSet(sys,Rstart,params,options);
    timeInt.set{i} = outputSet(sys,Rnext.ti,params,options);
    timeInt.time{i} = interval(tVec(i),tVec(i+1));
    
    % save reachable set
    Rstart = Rnext.tp;
    
    % safety property check
    if isfield(options,'specification')
        [res,timeInt,timePoint] = checkSpecification(...
            options.specification,Rnext.ti,timeInt,timePoint,i);
        if ~res; return; end
    end
    
    % log information
    verboseLog(options.verbose,i,tVec(i),params.tStart,params.tFinal);
    
end

% compute output set of last set
timePoint.set{end} = outputSet(sys,Rstart,params,options);

% specification fulfilled
res = true;

% ------------------------------ END OF CODE ------------------------------
