function p_GO = computeGO(sys, x0, u_ref, n_k)
% computeGO - compute the parameters of a GO model
%
% Syntax:
%    p_GO = computeGO(sys, x0, u_ref, n_k)
%
% Inputs:
%           sys     system
%           x0      initial state
%           u_ref   reference input trajectory
%           n_k     number of time steps
%
% Outputs:
%           p_GO   struct with the GO parameters for a given
%                  reference trajectory, where    
%               p_GO.A{k}      matrix that describes the influence of the 
%                               initial state x(1) on the state x(k+1)
%               p_GO.B{k,j}    matrix that describes the influence of the  
%                               input u(j) on the state x(k+1)     
%               p_GO.F{k,j}    matrix that describes the influence of the  
%                               linearization error L(j) on the state x(k+1)       
%               p_GO.C{k}      matrix that describes the influence of the 
%                               initial state x(1) on the output y(k)  
%               p_GO.D{k,j}    matrix that describes the influence of the  
%                               input u(j) on the output y(k)     
%               p_GO.E{k,j}    matrix that describes the influence of the  
%                               linearization error L(j) on the output y(k)
%               p_GO.x         reference state trajectory 
%                                   dimensions: n_x x (n_k+1)
%               p_GO.u         reference input trajectory  
%                                   dimensions: n_u x n_k
%               p_GO.y         reference output trajectory  
%                                   dimensions: n_y x n_k
%
% References:
%    [1] L. Luetzow and M. Althoff, "Reachset-conformant System
%        Identification," arXiv, 2024. 
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: --

% Authors:       Laura Luetzow
% Written:       21-July-2023
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

if ~isempty(x0) && ~isempty(u_ref)
    n_s = size(u_ref,3);
    x_ref = zeros(sys.dim, n_k+1,n_s);
    y_ref = zeros(sys.nrOfOutputs, n_k,n_s);
    if isa(x0, 'contSet')
        x0 = center(x0);
    end
    x_ref(:,1,:) = x0;
else
    x_ref = [];
    y_ref = [];
end
if ~isempty(u_ref)
    n_k = min(size(u_ref,2),n_k);
end
p_GO.A = cell(n_k,1);
p_GO.B = cell(n_k,n_k-1);
p_GO.F = cell(n_k,n_k-1);
p_GO.C = cell(n_k,1);
p_GO.D = cell(n_k,n_k);
p_GO.E = cell(n_k,n_k);

% compute the linearized system matrices
if iscell(sys.A)
    %linear time-varying system
    for k = 1 : n_k

        if ~isempty(x0) && ~isempty(u_ref)
            % compute reference solution x_ref and y_ref
            x_ref(:,k+1,:) = pagemtimes(sys.A{k},x_ref(:,k,:)) + pagemtimes(sys.B{k},u_ref(:,k,:));
            y_ref(:,k,:) = pagemtimes(sys.C{k},x_ref(:,k,:)) + pagemtimes(sys.D{k},u_ref(:,k,:));
        end

        % compute transfer matrices G for the x0->y(i) equation
        A_prod = eye(size(sys.A{1},1));
        for j = 1 : k-1
            A_prod = A_prod * sys.A{k-j};
            A_prod_j = 1;
            for i = 1 : k-j-1
                A_prod_j = A_prod_j * sys.A{k-i};
            end
            AA_prod = sys.A{k} * A_prod_j;
            p_GO.B{k,j} = AA_prod * sys.B{j};
            p_GO.F{k,j} = AA_prod * [eye(sys.dim) zeros(sys.dim, sys.nrOfOutputs)];

            CA_prod = sys.C{k} * A_prod_j;
            p_GO.D{k,j} = CA_prod * sys.B{j};
            p_GO.E{k,j} = CA_prod * [eye(sys.dim) zeros(sys.dim, sys.nrOfOutputs)];
        end
        p_GO.A{k} = sys.A{k} * A_prod;
        p_GO.B{k,k} = sys.B{k};
        p_GO.F{k,k} = [eye(sys.dim) zeros(sys.dim, sys.nrOfOutputs)]; % L = [L_x; L_y]

        p_GO.C{k} = sys.C{k} * A_prod;
        p_GO.D{k,k} = sys.D{k};
        p_GO.E{k,k} = [zeros(sys.nrOfOutputs, sys.dim) eye(sys.nrOfOutputs)]; % L = [L_x; L_y]
    end
else
    %linear time-invariant system    
    for k = 1 : n_k

        if ~isempty(x0) && ~isempty(u_ref)
            % compute reference solution x_ref and y_ref
            x_ref(:,k+1,:) = pagemtimes(sys.A,x_ref(:,k,:)) + pagemtimes(sys.B,u_ref(:,k,:));
            y_ref(:,k,:) = pagemtimes(sys.C,x_ref(:,k,:)) + pagemtimes(sys.D,u_ref(:,k,:));
        end

        % alternative computation
        if k>1
            p_GO.A{k} = sys.A * p_GO.A{k-1};
            p_GO.C{k} = sys.C * p_GO.A{k-1};
        else
            p_GO.A{k} = sys.A;
            p_GO.C{k} = sys.C;
        end
        p_GO.B{k,k} = sys.B;
        p_GO.D{k,k} = sys.D;
        for j = k-1 : -1: 1
            p_GO.B{k,j} = p_GO.A{k-j} * sys.B;
            p_GO.D{k,j} = sys.C* p_GO.B{k,j+1};
        end

    end
end

p_GO.x = x_ref;
p_GO.y = y_ref;
p_GO.u = u_ref;
end

% ------------------------------ END OF CODE ------------------------------
