function sys = nonlinearSysDT(sys)
% nonlinearSysDT - convert nonlinearARX sys to nonlinearSysDT
%
% Syntax:
%    sys = nonlinearSysDT(sys,dt)
%
% Description:
%    Converts a discrete-time linear system to a equivalent discrete-time 
%    linear ARX system (the process and measurement disturbances w and v
%    of the linear system will be combined with the input u to u_ARX =
%    [u; w; v])
%
% Inputs:
%    sys - discrete-time linear system (class: linearSysDT)
%    dt - sampling time
%
% Outputs:
%    sys - discrete-time linear ARX system (class: linearARX)
%
% Example:
%     % discrete-time system
%     f = @(y,u) [0.5*y(1,1) + u(1,1) - cos(u(2,1)); ...
%         0.4*y(3,1) + u(2,1)*cos(y(1,1)); 0.6*y(5,1) + u(4,1)*sin(y(1,1))];
%     dt = 0.1;
%     dim_y = 3;
%     dim_u = 2;
%     p = 2;
%     sys = nonlinearARX(f,dt,dim_y, dim_u, p);
% 
%     % convert to nonlinearSysDT
%     sys_DT = nonlinearSysDT(sys);


% See also: nonlinearSysDT, nonlinearARX

% Authors:       Laura Luetzow
% Written:       13-November-2023
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

dim_x = sys.nrOfOutputs * sys.dim;
dim_u = sys.nrOfInputs * (sys.dim + 1);
name = sprintf('%s_conv', sys.name);
fun = @(x,u) [x(sys.nrOfOutputs+1:end); sys.mFile(x,u)];
sys = nonlinearSysDT(name,fun,sys.dt,dim_x,dim_u);
    
end

% ------------------------------ END OF CODE ------------------------------
