function han = plot(SpS,varargin)
% plot - plots a projection of a spectrahedral shadow
%
% Syntax:
%    spec = plot(SpS)
%    spec = plot(SpS,dims)
%    spec = plot(SpS,dims,type)
%
% Inputs:
%    SpS - spectraShadow object 
%    dims - (optional) dimensions for projection
%    type - (optional) plot settings (LineSpec and Name-Value pairs),
%        including added pairs:
%          <'Splits',splits> - number of splits for refinement
%          <'ApproxType',approxType> - 'inner' (default) yields an
%          under-approximative plot, while 'outer' yields an
%          over-approximative plot of the spectrahedral shadow
%
% Outputs:
%    han - handle to the graphics object
%
% Example:
%    A0 = eye(4);
%    A1 = [-1 0 0 0;0 1 0 0;0 0 0 0;0 0 0 0];
%    A2 = [0 0 0 0;0 0 0 0;0 0 -1 0;0 0 0 1];
%    SpS = spectraShadow([A0 A1 A2]);
%    plot(SpS)
%
% Other m-files required: spectraShadow.m
% Subfunctions: none
% MAT-files required: none
%
% See also: none

% Authors:       Adrian Kulmburg
% Written:       02-August-2023
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

% 1. parse input
[SpS,dims,NVpairs,splits,approxType] = aux_parseInput(SpS,varargin{:});

% 2. preprocess; store result into a polytope, which will be the one to
% plot afterwards
P = aux_preprocess(SpS,dims,splits,approxType);

% 3. plot the polytope
han = aux_plot(P,dims,NVpairs{:});

% 4. clear han
if nargout == 0
    clear han;
end

end


% Auxiliary functions -----------------------------------------------------

function [SpS,dims,NVpairs,splits,approxType] = aux_parseInput(SpS,varargin)
    % parse input

    % default values for the optional input arguments
    dims = setDefaultValues({[1,2]},varargin);
    
    % read additional name-value pairs
    NVpairs = readPlotOptions(varargin(2:end));
    % read out 'Splits', default value given
    [NVpairs,splits] = readNameValuePair(NVpairs,'Splits','isscalar',100);
    % read out 'approxType', default value given
    [NVpairs,approxType] = readNameValuePair(NVpairs,'ApproxType','ischar','inner');
    
    % check input arguments
    inputArgsCheck({{SpS,'att','spectraShadow'};
        {dims,'att','numeric',{'nonempty','integer','positive','vector'}}; ...
        {splits,'att','numeric',{'scalar','integer','nonnegative'}} ...
        
    });
    inputArgsCheck({{approxType,'str',{'outer','inner'}}});
    
    % check dimension
    if length(dims) < 1
        throw(CORAerror('CORA:plotProperties',1));
    elseif length(dims) > 3
        throw(CORAerror('CORA:plotProperties',3));
    end
end


function P = aux_preprocess(SpS,dims,splits,approxType)

if representsa(SpS,'emptySet')
    P = polytope();
    return
end

% Let's start with the 1-Dim case: Transform it to the 2D case, and let
% that part of the algorithm do the heavy lifting
if length(dims) == 1
    % Project the spectrahedral shadow
    SpS = project(SpS,dims);
    % Take the Cartesian product with {0}
    SpS = cartProd(SpS, spectraShadow([0 0 1 0; 0 0 0 -1]));
    P = aux_preprocess(SpS,[dims 2],splits,approxType);
    return
elseif length(dims) == 2
    %Project the spectrahedral shadow
    SpS = project(SpS,dims);
    
    was_unbounded = false;
    % check if spectrahedral shadow is bounded
    if ~isBounded(SpS)
        % If it's unbounded, intersect it with current axis box
        was_unbounded = true;
    
        % get size of current plot
        [xLim,yLim,~] = getUnboundedAxisLimits();
        
        I_figure = interval([xLim(1);yLim(1)],[xLim(2);yLim(2)]);
        
        SpS = SpS & spectraShadow(I_figure);
    end
    
    c = center(SpS);
    SpS = SpS - c;
    
    
    % We uniformly distribute points on the unit circle
    phi = linspace(0, 2*pi, splits);
    x = cos(phi);
    y = sin(phi);
    
    % Now, it depends if we want an outer approx -> H-poly, or an inner
    % approx -> V-poly
    if strcmp(approxType,'outer')
        % Step by step, we construct the H-polytope encircling the
        % spectrahedral shadow
        P_A = [];
        P_b = [];
        for i=1:splits
            dir = [x(i);y(i)];
            [val,~] = SpS.supportFunc(dir);
            if val ~= Inf && val~= -Inf
                P_A = [P_A; [x(i) y(i)]];
                P_b = [P_b;val];
            end
        end
        if ~isempty(P_A)
            P = polytope(P_A,P_b);
            P = P + c;
            if was_unbounded
                % Just to be sure, to avoid small inaccuracies
                
                % get size of current plot
                [xLim,yLim,~] = getUnboundedAxisLimits(vertices(P));
        
                I_figure = interval([xLim(1);yLim(1)],[xLim(2);yLim(2)]);
                
                P = P & polytope(I_figure);
            end
        else
            P = polytope([],[]);
        end
        
    elseif strcmp(approxType,'inner')
        % Step by step, we construct the V-polytope inside the
        % spectrahedral shadow
        V = [];
        for i=1:splits
            dir = [x(i);y(i)];
            [~,v] = SpS.supportFunc(dir);
            if ~isempty(v)
                V = [V v];
            end
        end

        if ~isempty(V)
            Vt = V';
            k = convhulln(Vt);
            V = [Vt(k,1)';Vt(k,2)'];

            P = polytope(V);
            P = P + c;
            if was_unbounded
                % Just to be sure, to avoid small inaccuracies
                % get size of current plot
                [xLim,yLim,~] = getUnboundedAxisLimits(V);
        
                I_figure = interval([xLim(1);yLim(1)],[xLim(2);yLim(2)]);
                P = P & polytope(I_figure);
            end
        else
            P = polytope([],[]);
        end
    else
        throw(CORAerror('CORA:specialError',...
            'You chose a wrong argument for ApproxType; still, this error should not happen...'))
    end
elseif length(dims) == 3
    % Basically, this is the same as the dim=2 case, apart from some minor
    % adaptations
    %Project the spectrahedral shadow
    SpS = project(SpS,dims);
    
    was_unbounded = false;
    % check if spectrahedral shadow is bounded
    if ~isBounded(SpS)
        was_unbounded = true;
        % If it's unbounded, intersect it with current axis box
    
        % get size of current plot
        [xLim,yLim,zLim] = getUnboundedAxisLimits();
        
        I_figure = interval([xLim(1);yLim(1);zLim(1)],[xLim(2);yLim(2);zLim(2)]);
        
        SpS = SpS & spectraShadow(I_figure);
    end
    
    % Temporary measure; once the method 'center' for spectrahedra has been
    % implemented, replace c with that
    I = interval(SpS);
    c = center(I);
    SpS = SpS - c;
    
    
    % We uniformly distribute points on the unit sphere
    points_on_sphere = eq_point_set(3-1,splits);
    
    % Now, it depends if we want an outer approx -> H-poly, or an inner
    % approx -> V-poly
    if strcmp(approxType,'outer')
        % Step by step, we construct the H-polytope encircling the
        % spectrahedral shadow
        P_A = [];
        P_b = [];
        for i=1:splits
            dir = points_on_sphere(:,i);
            [val,~] = SpS.supportFunc(dir);
            if val ~= Inf && val~= -Inf
                P_A = [P_A;dir'];
                P_b = [P_b;val];
            end
        end
        if ~isempty(P_A)
            P = polytope(P_A,P_b);
            P = P + c;
            
            if was_unbounded
                % Just to be sure, to avoid small inaccuracies
                % get size of current plot
                [xLim,yLim,zLim] = getUnboundedAxisLimits(vertices(P));
        
                I_figure = interval([xLim(1);yLim(1);zLim(1)],[xLim(2);yLim(2);zLim(2)]);
                P = P & polytope(I_figure);
            end
        else
            P = polytope([],[]);
        end
        
    elseif strcmp(approxType,'inner')
        % Step by step, we construct the V-polytope inside the
        % spectrahedral shadow
        V = [];
        for i=1:splits
            dir =  points_on_sphere(:,i);
            [~,v] = SpS.supportFunc(dir);
            if ~isempty(v)
                V = [V v];
            end
        end

        if ~isempty(V)
            Vt = V';
            k = convhulln(Vt);
            V = [Vt(k,1)';Vt(k,2)';Vt(k,3)'];

            P = polytope(V);
            P = P + c;
            
            if was_unbounded
                % Just to be sure, to avoid small inaccuracies
                % get size of current plot
                [xLim,yLim,zLim] = getUnboundedAxisLimits(V);
        
                I_figure = interval([xLim(1);yLim(1);zLim(1)],[xLim(2);yLim(2);zLim(2)]);
                P = P & polytope(I_figure);
            end
        else
            P = polytope([],[]);
        end
    else
        throw(CORAerror('CORA:specialError',...
            'You chose a wrong argument for ApproxType; still, this error should not happen...'))
    end
end
end

function han = aux_plot(P,dims,varargin)
    if isemptyobject(P)
        han = plot(P,dims,varargin{:});
        % This will always output an error, as intended
    else
        han = plot(P,1:dim(P),varargin{:});
    end
end

% ------------------------------ END OF CODE ------------------------------
