function SpS_out = plus(SpS, summand)
% plus - Overloaded '+' operator for the minkowski addition of a
%    spectrahedral shadow and a vector, or a spectrahedral shadow and a
%    contSet object
%
% Syntax:
%    spec = plus(SpS,obj)
%
% Inputs:
%    SpS - spectraShadow object
%    obj - contSet object or numerical vector
%
% Outputs:
%    SpS_out - spectraShadow object, result of the addition of SpS and obj
%
% Example:
%    SpS = spectraShadow([1 0 1 0;0 1 0 -1]);
%    SpS_times_2 = SpS + SpS;
%    SpS_translated = SpS + 5;
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: none

% Authors:       Maximilian Perschl, Adrian Kulmburg
% Written:       24-April-2023
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

[SpS,summand] = findClassArg(SpS,summand,'spectraShadow');

if ~isnumeric(summand) && ~isa(summand,"spectraShadow")
    try
        summand = spectraShadow(summand);
    catch ME
        throw(CORAerror('CORA:noExactAlg',SpS,summand));
    end
end

% check dimensions
equalDimCheck(SpS,summand);

% addition between spectrahedral shadow and numerical vector
if isnumeric(summand)
    % get parameters of S
    A = SpS.A;
    G = SpS.G;
    c= SpS.c;
    
    % compute parameters of new spectrahedron
    if isemptyobject(SpS)
        c_new = [];
    else
        c_new = summand+c;
    end

    % construct spectrahedral shadow
    SpS_out = spectraShadow(A,c_new,G);
    
    SpS_out.bounded.val = SpS.bounded.val;
    SpS_out.fullDim.val = SpS.fullDim.val;
    SpS_out.emptySet.val = SpS.emptySet.val;
    
    if ~isempty(SpS.center.val)
        SpS_out.center.val = SpS.center.val + summand;
    end

% minkowski addition between two spectrahedra
else

    c_1 = SpS.c;
    c_2 = summand.c;

    G_1 = SpS.G;
    G_2 = summand.G;
    
    [A0_1,Ai_1] = getCoeffMatrices(SpS);
    [A0_2,Ai_2] = getCoeffMatrices(summand);
    
    m_1 = size(G_1,2);
    m_2 = size(G_2,2);
    k_1 = size(A0_1,1);
    k_2 = size(A0_2,1);
    
    c = c_1 + c_2;
    G = [G_1 G_2];
    A0 = blkdiag(A0_1,A0_2);
    Ai = cell([1 m_1+m_2]);
    for i=1:m_1
        Ai{i} = blkdiag(Ai_1{i},sparse(k_2,k_2));
    end
    for i=1:m_2
        Ai{m_1+i} = blkdiag(sparse(k_1,k_1),Ai_2{i});
    end
    
    SpS_out = spectraShadow([A0 cat(2, Ai{:})],c,G);
    
    % Adding some extra information
    if ~isempty(SpS.bounded.val) && ~isempty(summand.bounded.val)
        if SpS.bounded.val && summand.bounded.val
            SpS_out.bounded.val = true;
        else
            SpS_out.bounded.val = false;
        end
    end
    if (~isempty(SpS.fullDim.val) && SpS.fullDim.val) ...
            || (~isempty(summand.fullDim.val) && summand.fullDim.val)
        SpS_out.fullDim.val = true;
    end
    if (~isempty(SpS.emptySet.val) && SpS.emptySet.val) ...
            || (~isempty(summand.emptySet.val) && summand.emptySet.val)
        SpS_out.emptySet.val = true;
    elseif (~isempty(SpS.emptySet.val) && ~SpS.emptySet.val) ...
            && (~isempty(summand.emptySet.val) && ~summand.emptySet.val)
        SpS_out.emptySet.val = false;
    end
    
    if ~isempty(SpS.center.val) && ~isempty(summand.center.val)
        SpS_out.center.val = SpS.center.val + summand.center.val;
    end
end


% ------------------------------ END OF CODE ------------------------------
