classdef (Abstract) ASCIItable
% ASCIItable - creates a nice ASCII table
%
% Syntax:
%    table = ASCIItable(hvalues,formats,tableoptions)
%
% Inputs:
%    hvalues - cell array containing headings
%    formats - cell array containing cell formats
%    tableoptions - name-value pair
%        <'tbhline',tbhline> - char, top boundary horizontal line
%        <'tbhcorner',tbhcorner> - char, top heading boundary corner
%        <'tbhsep',tbhsep> - char, top heading boundary separator
%        <'mbhline',tbhline> - char, mid boundary horizontal line
%        <'mbhcorner',tbhcorner> - char, mid heading boundary corner
%        <'mbhsep',mbhsep> - char, mid heading boundary separator
%        <'bbhline',tbhline> - char, bottom boundary horizontal line
%        <'bbhcorner',tbhcorner> - char, bottom heading boundary corner
%        <'bbhsep',bbhsep> - char, bottom heading boundary separator
%        <'hbvline',hbvline> - char, heading boundary vertical line
%        <'hsep',hsep> - char, heading separator
%        <'cbvline',cbvline> - content boundary vertical line
%        <'csep',csep> - char, content separator
%        <'tpre',tbegin> - char, pre table text
%        <'tpost',tbegin> - char, post table text
%        <'tlpre',tbegin> - char, pre top line text
%        <'mlpre',tbegin> - char, pre mid line text
%        <'blpre',tbegin> - char, pre bottom line text
%        <'clpre',tbegin> - char, pre content line text
%        <'tlpost',tbegin> - char, post top line text
%        <'mlpost',tbegin> - char, post mid line text
%        <'blpost',tbegin> - char, post bottom line text
%        <'clpost',tbegin> - char, post content line text
%
% Outputs:
%    table - ASCIItable object
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: CORAtable

% Authors:       Lukas Koller, Tobias Ladner
% Written:       19-September-2024
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------


properties
    % main properties
    hvalues
    formats

    % boundary chars
    tbhline,tbhcorner,tbhsep
    mbhline,mbhcorner,mbhsep
    bbhline,bbhcorner,bbhsep
    % heading chars
    hbvline,hsep
    % content chars
    cbvline,csep
    % pre/post
    tpre,tpost
    tlpre,tlpost
    hlpre,hlpost
    mlpre,mlpost
    blpre,blpost
    clpre,clpost
end

methods
    function table = ASCIItable(hvalues,formats,varargin)
        % 1. parse input
        [hvalues,formats, ...
            tbhline,tbhcorner,tbhsep, ...
            mbhline,mbhcorner,mbhsep, ...
            bbhline,bbhcorner,bbhsep, ...
            hbvline,hsep, ...
            cbvline,csep, ...
            tpre,tpost, ...
            tlpre,tlpost, ...
            hlpre,hlpost, ...
            mlpre,mlpost, ...
            blpre,blpost, ...
            clpre,clpost] = aux_parseInput(hvalues,formats,varargin);

        % 2. save properties
        table.hvalues = hvalues;
        table.formats = formats;
        % boundary chars
        table.tbhline = tbhline;
        table.tbhcorner = tbhcorner;
        table.tbhsep = tbhsep;
        table.mbhline = mbhline;
        table.mbhcorner = mbhcorner;
        table.mbhsep = mbhsep;
        table.bbhline = bbhline;
        table.bbhcorner = bbhcorner;
        table.bbhsep = bbhsep;
        % heading chars
        table.hbvline = hbvline;
        table.hsep = hsep;
        % heading chars
        table.cbvline = cbvline;
        table.csep = csep;
        % pre/post
        table.tpre = tpre;
        table.tpost = tpost;
        table.tlpre = tlpre;
        table.tlpost = tlpost;
        table.hlpre = hlpre;
        table.hlpost = hlpost;
        table.mlpre = mlpre;
        table.mlpost = mlpost;
        table.blpre = blpre;
        table.blpost = blpost;
        table.clpre = clpre;
        table.clpost = clpost;
    end

    function printHeading(table)
        % print heading block

        % build heading row
        t = table;
        hrow = [...
          t.hlpre t.hbvline ' ' ...
          strjoin(t.hvalues, [' ' t.hsep ' ']) ...
          ' ' t.hbvline t.hlpost ...
        ];

        % print
        fprintf(t.tpre)
        table.printTopBoundaryRow();
        disp(hrow);
        table.printMidBoundaryRow();
    end
 
    function printContentRow(table,cvalues)
        % print content row

        % build content row
        t = table;
        crow = [ ...
            ... % start row with
            t.clpre t.cbvline ' ' ...
            ... % join all cells
            strjoin( ... 
                ... % build cell with correct format
                cellfun(@sprintf, ...
                    ... % build format with correct cell length
                    "%" + cellfun(@strlength,t.hvalues) + ...
                    t.formats, ...
                    ... % sprintf the content
                    cvalues,'UniformOutput',false), ...
                ... % separate each cell with
                [' ' t.csep ' ']) ...
            ... % and finish with
            ' ' t.cbvline t.clpost ...
        ];

        % print
        disp(crow);        
    end
    
    function printBottom(table)
        % print bottom of table
        table.printBottomBoundaryRow();
        fprintf(table.tpost)
        disp(' ')
    end

    % helper ---

    function printTopBoundaryRow(table)
        disp(buildBoundaryRow(table,table.tbhline,table.tbhcorner,table.tbhsep,table.tlpre,table.tlpost));
    end

    function printMidBoundaryRow(table)
        disp(buildBoundaryRow(table,table.mbhline,table.mbhcorner,table.mbhsep,table.mlpre,table.mlpost));
    end

    function printBottomBoundaryRow(table)
        disp(buildBoundaryRow(table,table.bbhline,table.bbhcorner,table.bbhsep,table.blpre,table.blpost));
    end
end

methods (Access=protected)
    function hbrow = buildBoundaryRow(table,bhline,bhcorner,bhsep,lpre,lpost)
        % build heading boundary row
        t = table;
        hbrow = [ ...
            ... % start row with
            lpre bhcorner bhline ...
            ... % join all cells
            strjoin( ...
                ... % build boundary with correct cell length
                cellfun( ...
                    @(s) repmat(bhline,[1,strlength(s)]), ...
                    t.hvalues, ...
                    'UniformOutput',false), ...
                ... % separate each cell with
                [bhline bhsep bhline]) ...
            ... % and finish with
            bhline bhcorner lpost...
        ];
    end
end

end


% Auxiliary functions -----------------------------------------------------

function [hvalues,formats, ...
            tbhline,tbhcorner,tbhsep, ...
            mbhline,mbhcorner,mbhsep, ...
            bbhline,bbhcorner,bbhsep, ...
            hbvline,hsep, ...
            cbvline,csep, ...
            tpre,tpost, ...
            tlpre,tlpost, ...
            hlpre,hlpost, ...
            mlpre,mlpost, ...
            blpre,blpost, ...
            clpre,clpost] = aux_parseInput(hvalues,formats,NVpairs)

    % parse input
    inputArgsCheck({ ...
        {hvalues,'att','cell','isrow'}; ...
        {formats,'att','cell','isrow'}; ...
    })

    if CHECKS_ENABLED
        % length of hvalues and formats must match
        if numel(hvalues) ~= numel(formats)
            throw(CORAerror('CORA:wrongValue','first/second','Length of headings and formats must match.'))
        end

        % check if no format has '%' (gets added in printContentRow to determine cell width)
        if any(cellfun(@(format) contains(format,'%'),formats,'UniformOutput',true))
            throw(CORAerror('CORA:wrongValue','second','Specify formats without leading ''%''. This is required to set the cell width correctly.'))
        end
    end

    % read name-value pairs
    % boundary chars
    [NVpairs,tbhline] = readNameValuePair(NVpairs,'tbhline','ischar','-');
    [NVpairs,tbhcorner] = readNameValuePair(NVpairs,'tbhcorner','ischar','+');
    [NVpairs,tbhsep] = readNameValuePair(NVpairs,'tbhsep','ischar',tbhline);
    [NVpairs,mbhline] = readNameValuePair(NVpairs,'mbhline','ischar',tbhline);
    [NVpairs,mbhcorner] = readNameValuePair(NVpairs,'mbhcorner','ischar',tbhcorner);
    [NVpairs,mbhsep] = readNameValuePair(NVpairs,'mbhsep','ischar',mbhline);
    [NVpairs,bbhline] = readNameValuePair(NVpairs,'bbhline','ischar',tbhline);
    [NVpairs,bbhcorner] = readNameValuePair(NVpairs,'bbhcorner','ischar',tbhcorner);
    [NVpairs,bbhsep] = readNameValuePair(NVpairs,'bbhsep','ischar',bbhline);
    % harding chars
    [NVpairs,hbvline] = readNameValuePair(NVpairs,'hbvline','ischar','|');
    [NVpairs,hsep] = readNameValuePair(NVpairs,'hsep','ischar','|');
    % content chars
    [NVpairs,cbvline] = readNameValuePair(NVpairs,'cbvline','ischar',hbvline);
    [NVpairs,csep] = readNameValuePair(NVpairs,'csep','ischar',hsep);
    % pre/post
    [NVpairs,tpre] = readNameValuePair(NVpairs,'tpre','ischar','');
    [NVpairs,tpost] = readNameValuePair(NVpairs,'tpost','ischar','');
    [NVpairs,tlpre] = readNameValuePair(NVpairs,'tlpre','ischar','');
    [NVpairs,tlpost] = readNameValuePair(NVpairs,'tlpost','ischar','');
    [NVpairs,hlpre] = readNameValuePair(NVpairs,'hlpre','ischar',tlpost);
    [NVpairs,hlpost] = readNameValuePair(NVpairs,'hlpost','ischar','');
    [NVpairs,mlpre] = readNameValuePair(NVpairs,'mlpre','ischar','');
    [NVpairs,mlpost] = readNameValuePair(NVpairs,'mlpost','ischar',tlpost);
    [NVpairs,blpre] = readNameValuePair(NVpairs,'blpre','ischar','');
    [NVpairs,blpost] = readNameValuePair(NVpairs,'blpost','ischar',tlpost);
    [NVpairs,clpre] = readNameValuePair(NVpairs,'clpre','ischar',hlpre);
    [NVpairs,clpost] = readNameValuePair(NVpairs,'clpost','ischar',hlpost);

end

% ------------------------------ END OF CODE ------------------------------
