classdef CORAtable < ASCIItable
% CORAtable - creates a table in the well-beloved  CORA format
%
% Syntax:
%    table = CORAtable(design,headings,formats)
%
% Inputs:
%    design - char, 'single', 'double', 'modern', 'minimalistic', 
%       'ascii', 'latex'
%    hvalues - cell array containing headings
%    formats - cell array containing cell formats
%
% Outputs:
%    table - CORAtable object
%
% Example: 
%    % init table
%    hvalues = {'Heading 1','Heading 2'};
%    formats = {'s','s'};
%    table = CORAtable('single',hvalues,formats);
%    
%    % print table
%    table.printHeading()
%    table.printContentRow({'v11','v12'});
%    table.printContentRow({'v21','v22'});
%    table.printContentRow({'v31','v32'});
%    table.printBottom();
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: ASCIItable

% Authors:       Tobias Ladner
% Written:       19-September-2024
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------


properties
    % main properties
    design
end

methods
    function table = CORAtable(design,hvalues,formats,varargin)
        % 1. parse input
        [design,hvalues,formats,tableoptions] = aux_parseInput(design,hvalues,formats);

        % 2. call constructor
        table = table@ASCIItable(hvalues,formats,tableoptions{:},varargin{:});

        % 3. save properties
        table.design = design;
    end
end

end


% Auxiliary functions -----------------------------------------------------

function [design,hvalues,formats,tableoptions] = aux_parseInput(design,hvalues,formats)

    % check input args
    inputArgsCheck({ ...
        {design,'str',{'single', 'double', 'modern', 'minimalistic','ascii','latex'}};    ...
        {hvalues,'att','cell'};    ...
        {formats,'att','cell'};    ...
    })

    % load design present
    switch design
        case 'single'
            tableoptions = aux_getSingleTableOptions();
        case 'double'
            tableoptions = aux_getDoubleTableOptions();
        case 'modern'
            tableoptions = aux_getModernTableOptions();
        case 'minimalistic'
            tableoptions = aux_getMinimalisticTableOptions();
        case 'ascii'
            tableoptions = aux_getAsciiTableOptions();
        case 'latex'
            tableoptions = aux_getLatexTableOptions(formats);
        otherwise
            % should have been caught at inputArgsCheck
            throw(CORAerror('CORA:wrongValue','third','Unknown design.'))
    end
end

function tableoptions = aux_getSingleTableOptions()
    tableoptions = {
        'tbhline','-','tbhcorner','|','hbvline','|','hsep','|','csep','|'
    };
end

function tableoptions = aux_getDoubleTableOptions()
    tableoptions = {
        'tbhline','=','tbhcorner','‖','hbvline','‖','hsep','|'
    };
end

function tableoptions = aux_getModernTableOptions()
    tableoptions = {
        'tbhline','_','tbhcorner','','mbhline','-','mbhcorner','','bbhline','_','bbhcorner','', ...
        'hbvline','','hsep','|'
    };
end

function tableoptions = aux_getMinimalisticTableOptions()
    tableoptions = {
        'tbhline','-','tbhcorner','','hbvline','','hsep',' '
    };
end

function tableoptions = aux_getAsciiTableOptions()
    tableoptions = {
        'tbhline','-','tbhcorner','+','tbhsep','+','mbhsep','+','bbhsep','+','hbvline','|','hsep','|'
    };
end

function tableoptions = aux_getLatexTableOptions(formats)
    tpre = sprintf([ ...
        '%%%% requires ''booktabs'' package\n' ...
        '\\\\begin{table}\n' ...
        '\\\\centering\n' ...
        '\\\\caption{My caption}\n' ...
        '\\\\label{tab:my-label}\n' ...
        '\\\\begin{tabular}{%s}\n' ...
    ], strjoin(cellfun(@aux_format2latex, formats,'UniformOutput',false),' '));
    tpost = [ ...
        '\\end{tabular}\n' ...
        '\\end{table}' ...
    ];

    tableoptions = { ...
        'tbhline',' ','tbhcorner','','hbvline','','hsep','&', ...
        'tlpre','\toprule','mlpre','\midrule','blpre','\bottomrule','hlpost','\\' ...
        'tpre', tpre, 'tpost', tpost, ...
    };
end

function latexformat = aux_format2latex(format)
    if contains(format,'s')
        % align strings on the left
        latexformat = 'l'; 
    else
        % align numbers etc. on the right
        latexformat = 'r';
    end
end

% ------------------------------ END OF CODE ------------------------------
