function PZ = quadMapPolyZono(PZ1, PZ2, n, k, m, error_order)
% quadMapPolyZono - compute quadratic map of uncertain matrices
%
% Syntax:
%    res = nnHelper.quadMapPolyZono(PZ1, PZ2, n, k, m)
%
% Inputs:
%    PZ1 - first polyZonotope
%    PZ2 - second polyZonotope
%    n - first dimension of first matrix polyZontope 
%    k - second dimension of first matrix polyZonotope = first dimension of second matrix polyZontope
%    m - second dimension of second matrix polyZontope
%    
% Outputs:
%    PZ - resulting zonotope from quadratic map of input polyzonotopes
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: -

% Authors:       Tobias Ladner
% Written:       28-March-2024
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

if nargin < 5
    throw(CORAerror('CORA:notEnoughInputArgs',5));
elseif nargin < 6
    error_order = [];
end

% transform given sets to polynomial matrix zonotopes

% init
c1 = PZ1.c; G1 = PZ1.G; GI1 = PZ1.GI; h1 = size(G1,2); q1 = size(GI1, 2);
c2 = PZ2.c; G2 = PZ2.G; GI2 = PZ2.GI; h2 = size(G2,2); q2 = size(GI2, 2);

% reshape to respective matric
c1 = reshape(c1,n,k); 
G1 = reshape(G1,n,k,h1,1);
GI1 = reshape(GI1,n,k,q1,1);
c2 = reshape(c2,k,m); G2 = reshape(G2,k,m,1,h2); GI2 = reshape(GI2,k,m,1,q2); 

% compute respective matrix multiplications
c1c2 = c1 * c2;
G1c2 = pagemtimes(G1, c2);
c1G2 = pagemtimes(c1, G2);
G1G2 = pagemtimes(G1, G2);

GI1c2 = pagemtimes(GI1, c2);
c1GI2 = pagemtimes(c1, GI2);
GI1GI2 = pagemtimes(G1, GI2);

% reshape back to vector
c1c2 = reshape(c1c2,n*m,[]);
G1c2 = reshape(G1c2,n*m,[]);
c1G2 = reshape(c1G2,n*m,[]);
G1G2 = reshape(G1G2,n*m,[]);
GI1c2 = reshape(GI1c2,n*m,[]);
c1GI2 = reshape(c1GI2,n*m,[]);
GI1GI2 = reshape(GI1GI2,n*m,[]);


% compute output exponent matrix
id1 = PZ1.id; id2 = PZ2.id;
id = unique([id1;id2]);
% extend exponent matrices
E1 = (id1' == id) * PZ1.E;
E2 = (id2' == id) * PZ2.E;
% compute multiplication
if ~isempty(PZ1.E) && ~isempty(PZ2.E)
    % sum each column of E1 to E2 and concatenate
    E1E2 = reshape(E1 + reshape(E2,[],1,h2),[],h1*h2);
end

% compute final generator and exponent matrices
h = min(size(E1,2),size(E2,2));
if all(E1(:,1:h)==E2(:,1:h),'all') && all(id1(1:numel(id2)) == id2)
    % do optimization
    cG = [G1c2(:,1:h)+c1G2(:,1:h) G1c2(:,h+1:end) c1G2(:,h+1:end)];
    cE = [E1(:,1:h) E1(:,h+1:end) E2(:,h+1:end)];
else
    % let removeReundantExonents to its magic within constructor
    cG = [G1c2 c1G2];
    cE = [E1 E2];
end
GI = [GI1c2, c1GI2, GI1GI2];

% combine
PZ_lin = polyZonotope(c1c2,cG,GI,cE);
PZ_ho = polyZonotope(0*c1c2,G1G2,[],E1E2);
if ~isempty(error_order)
    PZ_ho = reduce(PZ_ho,'girard',error_order);
end

% add via exact plus to keep c*G generators in front
PZ = exactPlus(PZ_lin,PZ_ho);

end

% ------------------------------ END OF CODE ------------------------------
