function [c, G, id] = reduceLabelledZono(c, G, id, nrGen)
% reduceLabelledZono - reduce the number of generators of a polynomial zonotope, 
%    where we exploit that an interval remainder is added when reducing 
%    with Girards method
%
% Syntax:
%    [c, G, GI, E, id] = nnHelper.reduceLabelledZono(c, G, GI, E, id, id_, nrGen, S)
%
% Inputs:
%    c - center of zonotope
%    G - generators of zonotope
%    id - ids
%    nrGen - nrGen
%
% Outputs:
%    [c, G, id, d] - reduced labelled zonotope
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: -

% Authors:       Tobias Ladner
% Written:       22-September-2024
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

if nrGen < size(G,2)
    % extract dimensions
    N = length(c);
    P = size(G,2);
    order = nrGen/N;

    % number of generators that stay unreduced (N generators are added again
    % after reduction)
    K = max(0,floor(N*order - N));

    % check if it is necessary to reduce the order
    if P > N*order && K >= 0
        
        % calculate the length of the generator vectors with a special metric
        len = sum(G.^2,1);

        % determine the smallest generators (= generators that are removed)
        [~,indSmallest] = sort(len,'descend');
        indSmallest = indSmallest(K+1:end);
        
        % construct a zonotope from the generators that are removed
        Grem = G(:,indSmallest);
     
        % remove the generators that got reduced from the generator matrices
        G(:,indSmallest) = [];
        id(indSmallest(indSmallest <= numel(id))) = [];

        % box over-approximation as approx error
        d = sum(abs(Grem),2);
        D = diag(d);
        D = D(:,d>0);

        % add to genetator matrix
        G = [G D];
    end
end

end

% ------------------------------ END OF CODE ------------------------------
