import re
import ast

# -----------------------------
# Utility: Dynamic Programming for Knapsack
# -----------------------------
def dp_knapsack(items, capacity):
    """
    Computes the optimal total value for a 0-1 knapsack instance using dynamic programming.
    
    Parameters:
      items (list): A list of [value, weight] pairs.
      capacity (int): The maximum weight capacity of the knapsack.
    
    Returns:
      int: The maximum total value achievable.
    """
    dp = [0] * (capacity + 1)
    for value, weight in items:
        # iterate backwards so each item is only used once
        for w in range(capacity, weight - 1, -1):
            dp[w] = max(dp[w], dp[w - weight] + value)
    return dp[capacity]

# -----------------------------
# Validation Function for Knapsack Solutions (List of Lists Format)
# -----------------------------
def validate_list_of_lists_format(raw_input_items_str, generated_text, capacity=20, real_optimal_value=None):
    """
    Validates a knapsack solution generated by an LLM.
    
    The expected LLM output format is:
      "Solution: [(8, 1), (9, 4), (5, 8), (4, 7)]\n  Value: 8+9+5+4=26\n  Weight: 1+4+8+7=20<=20"
    
    Now accepts solutions with miscalculated values or weights as long as the computed solution
    satisfies the knapsack constraints.
    
    Returns:
      tuple: (is_feasible, message, computed_value, optimal_value)
    """


    # --- Convert string of list of list to Python list of lists ---
    try:
        # Ensure input_items is a list of lists of numbers
        input_items = ast.literal_eval(raw_input_items_str)
        if not isinstance(input_items, list) or \
           not all(isinstance(item, list) and len(item) == 2 and \
                   isinstance(item[0], (int, float)) and isinstance(item[1], (int, float)) \
                   for item in input_items):
            print(f"Warning: Parsed input_items is not in the expected format: {input_items}")
            # Handle malformed items, e.g., by skipping or setting to a known error state
            # For now, we'll let the validation catch it if it's truly problematic after parsing
    except (ValueError, SyntaxError, TypeError) as e:
        print(f"Error parsing input_items : '{raw_input_items_str}'. Error: {e}")
        
        
    try:
        # Split the output into lines and find required lines.
        lines = generated_text.splitlines()
        solution_line = None
        value_line = None
        weight_line = None
        
        # Look for lines containing the keywords regardless of prefix.
        for line in lines:
            clean_line = line.strip()
            if "Solution:" in clean_line and solution_line is None:
                idx = clean_line.find("Solution:")
                solution_line = clean_line[idx:]
            elif "Value:" in clean_line and value_line is None:
                idx = clean_line.find("Value:")
                value_line = clean_line[idx:]
            elif "Weight:" in clean_line and weight_line is None:
                idx = clean_line.find("Weight:")
                weight_line = clean_line[idx:]
        
        if solution_line is None:
            return False, "Missing 'Solution:' line.", 0, None
        if value_line is None:
            return False, "Missing 'Value:' line.", 0, None
        if weight_line is None:
            return False, "Missing 'Weight:' line.", 0, None

        # --- Parse the solution list ---
        sol_str = solution_line[len("Solution:"):].strip()
        try:
            # Safely parse the list of tuples.
            selected_items = ast.literal_eval(sol_str)
            # Convert each item in the parsed solution to a list if it's a tuple.
            selected_items = [list(item) if isinstance(item, tuple) else item for item in selected_items]
        except Exception as e:
            return False, f"Error parsing solution list: {e}", 0, None

        if not isinstance(selected_items, list):
            return False, f"Parsed solution is not a list.", 0, None

        # Compute totals from the parsed solution.
        computed_value = sum(item[0] for item in selected_items)
        computed_weight = sum(item[1] for item in selected_items)

        # --- Parse the declared total Value ---
        try:
            # Expected format: "Value: 8+9+5+4=26"
            declared_value_str = value_line.split("=")[-1].strip()
            declared_value = int(declared_value_str)
        except Exception as e:
            # If we can't parse the declared value, just use computed value
            declared_value = computed_value

        # --- Parse the declared Weight and capacity using regex ---
        try:
            # Extract the part after "Weight:".
            weight_line_content = weight_line[len("Weight:"):].strip()
            # Look for the declared weight after "=".
            declared_weight_match = re.search(r"=(\d+)", weight_line_content)
            if declared_weight_match:
                declared_weight = int(declared_weight_match.group(1))
            else:
                declared_weight = computed_weight

            # Look for the capacity after "<=" if available.
            declared_capacity_match = re.search(r"<=(\d+)", weight_line_content)
            if declared_capacity_match:
                declared_capacity = int(declared_capacity_match.group(1))
            else:
                declared_capacity = capacity
        except Exception as e:
            # If we can't parse the declared weight, just use computed weight
            declared_weight = computed_weight
            declared_capacity = capacity

        # --- Build messages for potential mismatches ---
        messages = []
        
        # Check value calculation
        if computed_value != declared_value:
            messages.append(f"Declared value ({declared_value}) was incorrect. Using computed value ({computed_value}).")

        # Check weight calculation
        if computed_weight != declared_weight:
            messages.append(f"Declared weight ({declared_weight}) does not match computed weight ({computed_weight}). Using computed weight.")

        # --- Check if the weight exceeds capacity ---
        effective_capacity = capacity  # Use the provided capacity parameter
        if computed_weight > effective_capacity:
            return False, f"Computed weight ({computed_weight}) exceeds capacity ({effective_capacity}).", computed_value, None

        # --- Verify that the selected items are available (0-1 knapsack) ---
        available_items = input_items.copy()
        for item in selected_items:
            if item in available_items:
                available_items.remove(item)
            else:
                return False, f"Item {item} is not available or used more than once.", computed_value, None

        # --- Compute the optimal solution using DP ---
        optimal_value = dp_knapsack(input_items, effective_capacity)

        # --- Properly calculate gap ---
        # if optimal_value > 0:  # Avoid division by zero
        #     gap = (optimal_value - computed_value) / optimal_value
        # else:
        #     gap = 0 if computed_value == 0 else float('inf')

        # --- Construct a validation message ---
        message_parts = [f"Feasible solution with computed value {computed_value} and weight {computed_weight} "
                       f"(capacity {effective_capacity})."]
        
        if messages:
            message_parts.append(" " + " ".join(messages))
            
        message = "".join(message_parts)
        
        return True, message, computed_value, optimal_value  # Return computed_value

    except Exception as e:
        return False, f"Exception during validation: {str(e)}", 0, None
    
def validate_accord_format(raw_input_items_str, generated_text, capacity=20, real_optimal_value=None):
    """
    Validates a knapsack solution in the accord format.
    
    Expected format:
    Solution:
    [[9, 2] -> value:0+9=9, weight:0+2=2<=20],
    [[8, 6] -> value:9+8=17, weight:2+6=8<=20],
    ...
    
    Total Value: 29
    Total Weight: 19<=20
    
    Modified to be lenient with calculation errors but still verify constraints.
    
    Returns:
      tuple: (is_feasible, message, computed_value, optimal_value
    """

    # --- Convert string of list of list to Python list of lists ---
    try:
        # Ensure input_items is a list of lists of numbers
        input_items = ast.literal_eval(raw_input_items_str)
        if not isinstance(input_items, list) or \
           not all(isinstance(item, list) and len(item) == 2 and \
                   isinstance(item[0], (int, float)) and isinstance(item[1], (int, float)) \
                   for item in input_items):
            print(f"Warning: Parsed input_items is not in the expected format: {input_items}")
            # Handle malformed items, e.g., by skipping or setting to a known error state
            # For now, we'll let the validation catch it if it's truly problematic after parsing
    except (ValueError, SyntaxError, TypeError) as e:
        print(f"Error parsing input_items string for : '{raw_input_items_str}'. Error: {e}")
        
    try:
        # Split the output into lines and extract solution lines
        lines = generated_text.strip().splitlines()
        
        # Find the solution section and summary lines
        solution_start = None
        for i, line in enumerate(lines):
            if "Solution:" in line:
                solution_start = i
                break
        
        if solution_start is None:
            return False, "Missing 'Solution:' marker.", 0, None
        
        # Extract solution lines up to Total Value or empty line
        solution_lines = []
        i = solution_start + 1
        while i < len(lines) and not (lines[i].strip().startswith("Total Value:") or not lines[i].strip()):
            if lines[i].strip():  # Skip empty lines
                solution_lines.append(lines[i].strip())
            i += 1
        
        # Find Total Value and Total Weight lines
        total_value_line = None
        total_weight_line = None
        for i in range(solution_start, len(lines)):
            if i < len(lines) and lines[i].strip().startswith("Total Value:"):
                total_value_line = lines[i].strip()
            elif i < len(lines) and lines[i].strip().startswith("Total Weight:"):
                total_weight_line = lines[i].strip()
        
        if not solution_lines:
            return False, "No solution items found.", 0, None
        
        if total_value_line is None:
            return False, "Missing 'Total Value:' line.", 0, None
        
        if total_weight_line is None:
            return False, "Missing 'Total Weight:' line.", 0, None
        
        # Parse each solution line to extract items
        selected_items = []
        calculation_errors = []
        
        for line in solution_lines:
            # Remove trailing commas and clean up
            line = line.rstrip(',').strip()
            
            # Extract the item - improved regex pattern
            item_match = re.search(r'\[\[(\d+),\s*(\d+)\]', line)
            if not item_match:
                calculation_errors.append(f"Invalid item format in line: {line}")
                continue
            
            try:
                value = int(item_match.group(1))
                weight = int(item_match.group(2))
                item = [value, weight]
                selected_items.append(item)
            except (ValueError, IndexError) as e:
                calculation_errors.append(f"Error parsing item values: {str(e)}")
                continue
            
            # Check for calculation errors but don't fail the validation
            try:
                # Check value calculation
                value_match = re.search(r'value:(\d+)\+(\d+)=(\d+)', line)
                if value_match:
                    prev_value = int(value_match.group(1))
                    item_value = int(value_match.group(2))
                    new_value = int(value_match.group(3))
                    
                    if item_value != value:
                        calculation_errors.append(f"Value mismatch in item: {item_value} vs {value}")
                    
                    if prev_value + item_value != new_value:
                        calculation_errors.append(f"Incorrect value addition: {prev_value}+{item_value}≠{new_value}")
                
                # Check weight calculation
                weight_match = re.search(r'weight:(\d+)\+(\d+)=(\d+)<=(\d+)', line)
                if weight_match:
                    prev_weight = int(weight_match.group(1))
                    item_weight = int(weight_match.group(2))
                    new_weight = int(weight_match.group(3))
                    
                    if item_weight != weight:
                        calculation_errors.append(f"Weight mismatch in item: {item_weight} vs {weight}")
                    
                    if prev_weight + item_weight != new_weight:
                        calculation_errors.append(f"Incorrect weight addition: {prev_weight}+{item_weight}≠{new_weight}")
            except Exception as e:
                calculation_errors.append(f"Error checking calculations: {str(e)}")
        
        # If no valid items were parsed, return an error
        if not selected_items:
            return False, "No valid items could be parsed from the solution.", 0, None
        
        # Compute totals from the selected items
        computed_value = sum(item[0] for item in selected_items)
        computed_weight = sum(item[1] for item in selected_items)
        
        # Extract declared totals
        try:
            total_value_match = re.search(r'Total Value:\s*(\d+)', total_value_line)
            declared_value = int(total_value_match.group(1)) if total_value_match else computed_value
            
            total_weight_match = re.search(r'Total Weight:\s*(\d+)<=(\d+)', total_weight_line)
            declared_weight = int(total_weight_match.group(1)) if total_weight_match else computed_weight
            declared_capacity = int(total_weight_match.group(2)) if total_weight_match else capacity
        except Exception as e:
            calculation_errors.append(f"Error parsing totals: {str(e)}")
            declared_value = computed_value
            declared_weight = computed_weight
            declared_capacity = capacity
        
        # Check totals but don't fail validation
        if declared_value != computed_value:
            calculation_errors.append(f"Declared value {declared_value} doesn't match computed value {computed_value}")
        
        if declared_weight != computed_weight:
            calculation_errors.append(f"Declared weight {declared_weight} doesn't match computed weight {computed_weight}")
        
        # Check weight constraint - this is critical for feasibility
        if computed_weight > capacity:
            return False, f"Weight {computed_weight} exceeds capacity {capacity}", computed_value, None
        
        # Verify that the selected items are available (0-1 knapsack)
        available_items = input_items.copy()
        for item in selected_items:
            if item in available_items:
                available_items.remove(item)
            else:
                return False, f"Item {item} is not available or used more than once.", computed_value, None
        
        # Compute the optimal solution using DP
        optimal_value = dp_knapsack(input_items, capacity)
        
        # Properly calculate gap with null check
        # if optimal_value > 0:  # Prevent division by zero
        #     gap = (optimal_value - computed_value) / optimal_value
        # else:
        #     gap = 0 if computed_value == 0 else float('inf')
        
        # Build the final message
        message_parts = [f"Feasible solution with value {computed_value} and weight {computed_weight} "
                       f"(capacity {capacity})."]
        
        if calculation_errors:
            message_parts.append(" Calculation errors: " + "; ".join(calculation_errors))
            
        message = "".join(message_parts)
        
        return True, message, computed_value, optimal_value
    
    except Exception as e:
        return False, f"Exception during accord validation: {str(e)}", 0, None



