#!/bin/bash
#SBATCH --job-name=coderelabel_postprocess
#SBATCH --partition=spyder
#SBATCH -c 64
#SBATCH -t 4-00:00
#SBATCH --mem=100G
#SBATCH --output=%j.out
#SBATCH --error=%j.err

# Function to display usage information
usage() {
    echo "Usage: $0 [-c <experiment_config_file>] [-p <part>]"
    echo "  -c <config>     Experiment-specific config file (optional)"
    echo "  -p <part>       Part number to process"
    exit 1
}

# Parse command line arguments
while getopts ":c:p:" opt; do
    case ${opt} in
        c )
            CONFIG_FILE=$OPTARG
            ;;
        p )
            PART=$OPTARG
            ;;
        \? )
            usage
            ;;
    esac
done

# Check if part number is provided
if [ -z "$PART" ]; then
    echo "Error: Part number (-p) must be specified"
    usage
fi

############################################################
# Source configuration files
############################################################
if [ -z "$CONFIG_FILE" ]; then
    CONFIG_FILE="exps/datasetgen/scripts/configs/dataset_relabel_config.sh"
fi

# Check if config file exists
if [ ! -f "$CONFIG_FILE" ]; then
    echo "Error: Experiment config file '$CONFIG_FILE' not found."
    exit 1
fi

source "$CONFIG_FILE"

############################################################
# Override paths with part number
INPUT_PATH="${RESPONSES_DIR}/${PROMPT_NAME}__${MODEL_NAME_SHORT}__${DATE}_part_${PART}.json"
OUTPUT_PATH="${POSTPROCESS_DIR}/${PROMPT_NAME}__${MODEL_NAME_SHORT}__${DATE}_part_${PART}.json"

# Print configuration
echo "Configuration:"
echo "- Config file: $CONFIG_FILE"
echo "- Part number: $PART"
echo "- Input path: ${INPUT_PATH}"
echo "- Output path: ${OUTPUT_PATH}"

# Check if output path already exists
if [ -f "${OUTPUT_PATH}" ]; then
    echo "Output file ${OUTPUT_PATH} already exists. Skipping postprocessing."
else
    # Use the part number as the display number
    DISPLAY_NUM=$((1000 + PART))
    Xvfb :${DISPLAY_NUM} -screen 0 1600x1200x16 &
    XVFB_PID=$!
    export DISPLAY=:${DISPLAY_NUM}.0

    echo "Starting postprocessing for part ${PART}"

    # Note: we reuse the `postprocess.py` from `codegen`
    ${PYTHON_TURTLE} src/turtlegfx_datagen/codegen/postprocess.py \
        --input_path "${INPUT_PATH}" \
        --output_path "${OUTPUT_PATH}" \
        --output_type success \
        --max_workers 48

    if [ $? -ne 0 ]; then
        echo "Error: Python script failed to execute properly."
        kill $XVFB_PID
        exit 1
    fi

    kill $XVFB_PID

    echo "RELABEL: Postprocessing complete for part ${PART}. Output saved to ${OUTPUT_PATH}"
fi
