"""
Usage: 
"""
from src.turtlegfx_datagen.prompts.build_prompts import build_prompts_from_json
import os
import argparse
from transformers import AutoTokenizer

def apply_prompt_template(item, prompt_template):
    """
    Apply the prompt template to the item with the given specs.

    Args:
        item (dict): A dictionary from the input JSON file.
        prompt_template (str): The template string with placeholders.

    Returns:
        prompt (str): The generated prompt.
    """
    prompt = prompt_template
    return prompt

def custom_function(item, prompt_template, input_file, **kwargs):
    question = apply_prompt_template(item, prompt_template)

    message = [
        {
            'role': 'system', 
            'content': "You are a helpful assistant."
        },
        {
            'role': 'user',
            'content': f"<image>\n{question}"
        }
    ]

    return [{
        "id": item.get('id', ''),
        "message": message,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_llavaov_chat(item, prompt_template, input_file, **kwargs):
    """
    LLaVA-OneVision chat custom function.
    Model: "llava-hf/llava-onevision-qwen2-7b-ov-chat-hf"
    """
    # Same as llavaov
    return custom_function_llavaov(item, prompt_template, input_file, **kwargs)

def custom_function_llavaov(item, prompt_template, input_file, **kwargs):
    """
    LLaVA-OneVision custom function.
    Model: "llava-hf/llava-onevision-qwen2-7b-ov-hf"
    """
    question = apply_prompt_template(item, prompt_template)

    # Following run_llava_onevision format for image
    prompt = f"<|im_start|>user <image>\n{question}<|im_end|> <|im_start|>assistant\n"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],

    }]

def custom_function_qwen2vl_instruct(item, prompt_template, input_file, **kwargs):
    """
    Qwen2-VL custom function
    Model: "Qwen/Qwen2-VL-7B-Instruct"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Following run_qwen2_vl format
    prompt = ("<|im_start|>system\nYou are a helpful assistant.<|im_end|>\n"
             "<|im_start|>user\n<|vision_start|><|image_pad|><|vision_end|>"
             f"{question}<|im_end|>\n"
             "<|im_start|>assistant\n")

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_nvlm(item, prompt_template, input_file, **kwargs):
    """
    NVLM-D custom function
    Model: "nvidia/NVLM-D-72B"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Following run_nvlm_d format
    messages = [{'role': 'user', 'content': f"<image>\n{question}"}]
    model_name = "nvidia/NVLM-D-72B"
    tokenizer = AutoTokenizer.from_pretrained(model_name, trust_remote_code=True)
    prompt = tokenizer.apply_chat_template(messages,
                                         tokenize=False,
                                         add_generation_prompt=True)

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_pixtralhf(item, prompt_template, input_file, **kwargs):
    """
    Pixtral-HF custom function
    Model: "mistral-community/pixtral-12b"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Construct the prompt using the format from run_pixtralhf
    prompt = f"<s>[INST]{question}\n[IMG][/INST]"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_pixtral(item, prompt_template, input_file, **kwargs):
    """
    Pixtral custom function
    Model: "mistralai/Pixtral-Large-Instruct-2411"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Following official Pixtral format
    messages = [
        {
            "role": "user",
            "content": [
                {
                    "type": "text",
                    "text": question
                },
                {
                    "type": "image_url",
                    "image_url": {
                        "url": item['task_image']
                    }
                }
            ]
        }
    ]

    return [{
        "id": item.get('id', ''),
        "messages": messages,  # Changed from prompt to messages
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_internvl2(item, prompt_template, input_file, **kwargs):
    """
    internvl2 custom function
    Model: "OpenGVLab/internvl2-2B"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Following run_internvl format
    model_name = "OpenGVLab/InternVL2-2B"
    tokenizer = AutoTokenizer.from_pretrained(model_name, trust_remote_code=True)
    messages = [{'role': 'user', 'content': f"<image>\n{question}"}]
    prompt = tokenizer.apply_chat_template(messages,
                                         tokenize=False,
                                         add_generation_prompt=True)

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]


def custom_function_gpt(item, prompt_template, input_file, **kwargs):
    """
    OpenAI custom function for generating prompts with images for batch processing.
    """
    question = apply_prompt_template(item, prompt_template)
    
    messages = [
        {
            'role': 'system',
            'content': "You are a helpful assistant."
        },
        {
            'role': 'user',
            'content': [
                {
                    'type': 'text',
                    'text': question
                },
                {
                    'type': 'image_url',
                    'image_url': {
                        'url': f"{item['task_image']}"
                    }
                }
            ]
        }
    ]

    return [{
        "id": item.get('id', ''),
        "message": messages,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_phi3v(item, prompt_template, input_file, **kwargs):
    """
    Phi-3-Vision custom function
    Model: "microsoft/Phi-3-vision-128k-instruct"
    """
    question = apply_prompt_template(item, prompt_template)
    
    prompt = f"<|user|>\n<|image_1|>\n{question}<|end|>\n<|assistant|>\n"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_blip2(item, prompt_template, input_file, **kwargs):
    """
    BLIP-2 custom function
    Model: "Salesforce/blip2-opt-2.7b"
    """
    question = apply_prompt_template(item, prompt_template)
    
    prompt = f"Question: {question} Answer:"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_chameleon(item, prompt_template, input_file, **kwargs):
    """
    Chameleon custom function
    Model: "facebook/chameleon-7b"
    """
    question = apply_prompt_template(item, prompt_template)
    
    prompt = f"{question}<image>"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_minicpmv(item, prompt_template, input_file, **kwargs):
    """
    MiniCPM-V custom function
    Model: "openbmb/MiniCPM-V-2_6"
    """
    question = apply_prompt_template(item, prompt_template)
    
    model_name = "openbmb/MiniCPM-V-2_6"
    tokenizer = AutoTokenizer.from_pretrained(model_name, trust_remote_code=True)
    messages = [{'role': 'user', 'content': f'(<image>./</image>)\n{question}'}]
    prompt = tokenizer.apply_chat_template(messages, tokenize=False, add_generation_prompt=True)

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_glm4v(item, prompt_template, input_file, **kwargs):
    """
    GLM-4v custom function
    Model: "THUDM/glm-4v-9b"
    """
    question = apply_prompt_template(item, prompt_template)
    
    prompt = question

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_idefics3(item, prompt_template, input_file, **kwargs):
    """
    Idefics3-8B-Llama3 custom function
    Model: "HuggingFaceM4/Idefics3-8B-Llama3"
    """
    question = apply_prompt_template(item, prompt_template)
    
    prompt = f"<|begin_of_text|>User:<image>{question}<end_of_utterance>\nAssistant:"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_paligemma(item, prompt_template, input_file, **kwargs):
    """
    PaliGemma custom function
    Model: "google/paligemma-3b-mix-224"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # prompt = "caption en"  # PaliGemma has special prompt format for VQA
    prompt = question

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_molmo(item, prompt_template, input_file, **kwargs):
    """
    Molmo custom function
    Model: "allenai/Molmo-7B-D-0924"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Molmo uses the question directly as the prompt
    prompt = question

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_aria(item, prompt_template, input_file, **kwargs):
    """
    ARIA custom function
    Model: "rhymes-ai/Aria"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Following run_aria format
    prompt = f"<|im_start|>user\n<fim_prefix><|img|><fim_suffix>\n{question}<|im_end|>\n<|im_start|>assistant\n"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

def custom_function_deepseek_vl2(item, prompt_template, input_file, **kwargs):
    """
    Deepseek-VL2 custom function
    Model: "deepseek-ai/deepseek-vl2-small"
    """
    question = apply_prompt_template(item, prompt_template)
    
    # Following run_deepseek_vl2 format
    prompt = f"<|User|>: <image>\n{question}\n\n<|Assistant|>:"

    return [{
        "id": item.get('id', ''),
        "prompt": prompt,
        "task_image": item['task_image'],
        "code": item['code'],
        "src_file": input_file,
        "parent_id": item['id'],
    }]

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Generate prompts from input JSON and template")
    parser.add_argument('--input_file', type=str, required=True, help='Path to input JSON file')
    parser.add_argument('--output_file', type=str, required=True, help='Path to output JSON file')
    parser.add_argument('--prompt_template', type=str, required=True, help='Prompt template string or file path')
    parser.add_argument('--model', type=str, required=True, help='Model name', default="llavaov")
    args = parser.parse_args()

    if os.path.isfile(args.prompt_template):
        with open(args.prompt_template, 'r') as f:
            prompt_template = f.read()
    else:
        prompt_template = args.prompt_template
    
    custom_functions = {
        "llavaov": custom_function_llavaov,
        "llavaov_chat": custom_function_llavaov_chat,
        "qwen2vl_instruct": custom_function_qwen2vl_instruct,
        "nvlm": custom_function_nvlm,
        "pixtral": custom_function_pixtral,
        "pixtralhf": custom_function_pixtralhf,
        "internvl2": custom_function_internvl2,
        "gpt": custom_function_gpt,
        "molmo": custom_function_molmo,
        "phi3v": custom_function_phi3v,
        "blip2": custom_function_blip2,
        "chameleon": custom_function_chameleon,
        "minicpmv": custom_function_minicpmv,
        "glm4v": custom_function_glm4v,
        "idefics3": custom_function_idefics3,
        "paligemma": custom_function_paligemma,
        "aria": custom_function_aria,
        "deepseek_vl2": custom_function_deepseek_vl2,
    }

    assert args.model in custom_functions.keys(), f"Model {args.model} not supported"
    
    build_prompts_from_json(
            input_file=args.input_file,
            output_file=args.output_file,
            prompt_template=prompt_template,
            custom_function=custom_functions[args.model],
            split=False,
        )