import os
import gc
import sys

from src.turtlegfx.utils.load_data import load_code
from src.turtlegfx_datagen.utils.extract_python import extract_draw_function
from src.turtlegfx.utils.base64img import convert_img_to_base64, convert_base64_to_img
from src.turtlegfx_datagen.utils.extract_python import extract_python_code_from_text
from src.turtlegfx.utils.code_modifier import remove_turtle_setup_calls

# Define error and success types
ERROR_NO_CODE_PROVIDED = 'ERROR_NO_CODE_PROVIDED'
ERROR_NO_FUNC_FOUND = 'ERROR_NO_FUNC_FOUND'
ERROR_EXEC = 'ERROR_EXEC'
ERROR_DRAW_FUNC_EXEC = 'ERROR_DRAW_FUNC_EXEC'
ERROR_OPEN_IMAGE = 'ERROR_OPEN_IMAGE'
ERROR_UNKNOWN = 'ERROR_UNKNOWN'
ERROR_EXTRACT_CODE = 'ERROR_EXTRACT_CODE'
ERROR_IMAGE_TOO_LARGE = 'ERROR_IMAGE_TOO_LARGE'
ERROR_ASPECT_RATIO_TOO_EXTREME = 'ERROR_ASPECT_RATIO_TOO_EXTREME'
ERROR_INVALID_SYNTAX = 'ERROR_INVALID_SYNTAX'
ERROR_INVALID_BOUNDING_BOX = 'ERROR_INVALID_BOUNDING_BOX'
EXEC_SUCCESS = 'EXEC_SUCCESS'

# Define maximum allowed dimensions and aspect ratio
MAX_WIDTH = 2000
MAX_HEIGHT = 2000
MAX_ASPECT_RATIO = 100

def _turtle_worker(code_str, show_screen, record_turtle_states=False):
    """
    Worker function to execute turtle code in a fresh environment.

    Args:
        code_str (str): The turtle code to execute.
        show_screen (bool): If True, displays the drawing window and shows the drawing animation.

    Returns:
        dict: A dictionary containing the execution result.
    """
    from turtle import TurtleScreen
    from src.turtlegfx.emulate.xturtle import XTurtle as RawTurtle
    from tkinter import Tk, Canvas
    import io
    from PIL import Image
    import math
    from math import pi, sqrt, tan, hypot, cos, radians, sin

    result = {
        "status": 'fail',
        "error_type": ERROR_UNKNOWN,
        "message": "Unknown error occurred."
    }

    try:
        # Save original stdout and redirect to null device
        old_stdout = sys.stdout
        sys.stdout = io.StringIO()  # or use open(os.devnull, 'w')

        # Step 1: Set up the screen and turtle
        try:
            root = Tk()
            if not show_screen:
                root.withdraw()  # Hide the root window if not showing the screen

            canvas = Canvas(master=root, width=2000, height=2000)
            canvas.pack()

            screen = TurtleScreen(canvas)
            screen.screensize(2000, 2000)  # Set a large canvas size

            turtle = RawTurtle(screen)
            turtle.hideturtle()

            if show_screen:
                turtle.speed(6)  # Set to a reasonable speed (1 slowest, 10 fastest)
                screen.tracer(1)  # Enable animation
            else:
                turtle.speed(0)  # Fastest drawing speed
                screen.tracer(0)  # Disable animation for instant drawing
        except Exception as e:
            result = {
                "status": 'fail',
                "error_type": ERROR_UNKNOWN,
                "message": f'Error setting up screen and turtle: {e}'
            }
            return result

        namespace = {
            'turtle': turtle, 
            'screen': screen, 
            'math': math,
            'sqrt': sqrt,
            'tan': tan,
            'hypot': hypot,
            'cos': cos,
            'radians': radians,
            'sin': sin,
            'pi': pi,
            'Turtle': RawTurtle
        }

        # Step 2: Extract the draw function from the code
        try:
            code_str = extract_draw_function(code_str)
        except Exception as e:
            result = {
                "status": 'fail',
                "error_type": ERROR_NO_FUNC_FOUND,
                "message": f'Error extracting draw function: {e}'
            }
            return result

        # Step 3: Execute the code
        try:
            exec(code_str, namespace)
        except Exception as e:
            result = {
                "status": 'fail',
                "error_type": ERROR_EXEC,
                "message": f'Error executing code: {e}'
            }
            return result

        # Step 4: Call the draw(turtle) function if it exists
        try:
            if 'draw' in namespace and callable(namespace['draw']):
                namespace['draw'](turtle)
            else:
                result = {
                    "status": 'fail',
                    "error_type": ERROR_NO_FUNC_FOUND,
                    "message": "No 'draw' function defined in the code."
                }
                return result
        except Exception as e:
            result = {
                "status": 'fail',
                "error_type": ERROR_DRAW_FUNC_EXEC,
                "message": f'Error executing draw function: {e}'
            }
            return result

        # Step 5: Update the screen
        try:
            screen.update()
        except Exception as e:
            result = {
                "status": 'fail',
                "error_type": ERROR_UNKNOWN,
                "message": f'Error updating screen: {e}'
            }
            return result

        # Step 6: Capture the canvas
        try:
            bbox = canvas.bbox("all")  # (min_x, min_y, max_x, max_y)

            if bbox is not None:
                min_x, min_y, max_x, max_y = bbox
            else:
                min_x = min_y = max_x = max_y = 0

            # Ensure min_x, min_y, max_x, max_y are valid
            if min_x == max_x or min_y == max_y:
                result = {
                    "status": 'fail',
                    "error_type": ERROR_INVALID_BOUNDING_BOX,
                    "message": 'Bounding box is invalid or empty.'
                }
                return result

            padding = 20
            min_x -= padding
            min_y -= padding
            max_x += padding
            max_y += padding

            width = max_x - min_x
            height = max_y - min_y

            # Check for image size
            if width > MAX_WIDTH or height > MAX_HEIGHT:
                result = {
                    "status": 'fail',
                    "error_type": ERROR_IMAGE_TOO_LARGE,
                    "message": f'Image size too large: width={width}, height={height}'
                }
                return result

            # Check for aspect ratio
            if height != 0:
                aspect_ratio = width / height
            else:
                aspect_ratio = float('inf')

            if aspect_ratio > MAX_ASPECT_RATIO or aspect_ratio < 1 / MAX_ASPECT_RATIO:
                result = {
                    "status": 'fail',
                    "error_type": ERROR_ASPECT_RATIO_TOO_EXTREME,
                    "message": f'Aspect ratio too extreme: width={width}, height={height}, aspect_ratio={aspect_ratio}'
                }
                return result

            # Update the canvas scroll region
            canvas.config(scrollregion=(min_x, min_y, max_x, max_y))
            canvas.config(width=width, height=height)
            canvas.update()

            # Save the drawing to an in-memory buffer
            ps_data = canvas.postscript(
                colormode='color',
                x=min_x,
                y=min_y,
                width=width,
                height=height,
                pagewidth=width,
                pageheight=height,
            )

            # Convert postscript to base64 image
            img = Image.open(io.BytesIO(ps_data.encode('utf-8')))
            base64_img = convert_img_to_base64(img)
        except Exception as e:
            result = {
                "status": 'fail',
                "error_type": ERROR_UNKNOWN,
                "message": f'Error capturing canvas: {e}'
            }
            return result
        finally:
            # Restore stdout to its original value
            sys.stdout = old_stdout

            # save turtle states before cleaning up
            turtle_states = turtle.get_turtle_states()

            # Step 7: Clean up and exit
            if show_screen:
                root.mainloop()
            else:
                # Update and destroy the root window
                root.update_idletasks()
                root.update()
                root.destroy()

            # Delete references to GUI elements
            del root, canvas, screen, turtle

            gc.collect()

        # Step 8: Report success
        result = {
            "status": 'success',
            "error_type": EXEC_SUCCESS,
            "message": "Execution completed successfully.",
            "image": base64_img,
        }

        if record_turtle_states:
            result["turtle_states"] = turtle_states

    except Exception as e:
        result = {
            "status": 'fail',
            "error_type": ERROR_UNKNOWN,
            "message": f'An unknown error occurred: {e}'
        }

    return result

class Executor:
    def __init__(self):
        pass

    def run_model_output(self, model_output, show_screen=False, save_filename=None, record_turtle_states=False):
        try:
            extracted_code = extract_python_code_from_text(model_output)
        except Exception as e:
            return None, {
                "status": "fail",
                "error_type": ERROR_EXTRACT_CODE,
                "message": f"Error extracting code from model output: {e}"
            }
        return self.run(code=extracted_code, 
                       show_screen=show_screen, 
                       save_filename=save_filename, 
                       record_turtle_states=record_turtle_states)

    def run(self, code=None, show_screen=False, save_filename=None, record_turtle_states=False):
        """
        Executes the given turtle code.

        Args:
            code (str): The turtle code to execute. It can be a code string, a filename, or an identifier for external sources.
            show_screen (bool): If True, displays the drawing window and shows the drawing animation.
            save_filename (str): The filename to save the image to if specified.

        Returns:
            base64_image (str): The base64 encoded image if the execution was successful. None otherwise.
            result: A message dict containing the execution result.
        """

        if code is None:
            return None, {
                "status": "fail",
                "error_type": ERROR_NO_CODE_PROVIDED,
                "message": "No code provided."
            }

        code_str = self._load_code(code)
        
        # Add error handling for code cleanup
        try:
            code_str = remove_turtle_setup_calls(code_str)
        except Exception as e:
            return None, {
                "status": "fail",
                "error_type": ERROR_INVALID_SYNTAX,
                "message": f"Invalid Python syntax in code: {str(e)}"
            }

        # Run the code
        result = _turtle_worker(code_str, show_screen, record_turtle_states)

        # If there was an error, return
        if result["status"] != 'success':
            return None, result

        # Process the image
        if save_filename:
            img = convert_base64_to_img(result['image'])
            img.save(save_filename)
        image = result.pop('image')  # remove image from result
        return image, result

    def _load_code(self, code):
        """
        Loads code from various sources.

        Args:
            code (str): Code string, filename, or external code identifier.

        Returns:
            str: The code string.
        """
        if code.startswith('midi') or code.startswith('maxi'):
            # Placeholder for external code loading
            code_str = load_code(code)
        elif os.path.isfile(code):
            with open(code, 'r') as file:
                code_str = file.read()
        else:
            code_str = code
        return code_str

if __name__ == '__main__':
    code1 = '''
# Drawing a circle
def draw(t):
    t.circle(100)
'''

    code2 = '''
# Drawing a square
def draw(t):
    for _ in range(4):
        t.forward(100)
        t.right(90)
'''
    code3 = '''
# Drawing a star
def draw(t):
    for _ in range(5):
        t.forward(100)
        t.right(144)
'''

    code4 = '''
# Drawing a square with a red fill
def draw(t):
    t.fillcolor('red')
    t.begin_fill()
    for _ in range(4):
        t.forward(100)
        t.right(90)
    t.end_fill()
    '''

    code5 = '''
def draw(t):
    # Set the initial heading to face upwards
    t.setheading(90)

    # Define the bottom-left corner drawing function
    def draw_bottom_left_corner(t, size=100, degree=90):
        t.forward(size)
        t.left(degree)
        t.forward(size)
        t.left(degree)
        t.forward(size)
        t.right(degree)
        t.forward(size)
        t.left(degree)

    # Draw 3 bottom-left corners
    # fill the shape
    t.fillcolor('red')
    t.begin_fill()
    t.left(90)
    t.forward(100)
    t.left(90)
    t.forward(100)
    t.left(90)
    t.forward(200)
    t.left(90)
    t.forward(200)
    t.end_fill()

    # Move forward after drawing the corners
    t.backward(100)
    draw_bottom_left_corner(t, 100)
    draw_bottom_left_corner(t, 100)
    draw_bottom_left_corner(t, 100)
    draw_bottom_left_corner(t, 100)
'''
    code6 = '''
def draw(t):
    """
    Recreate the given geometric pattern on the screen.
    
    Args:
    t: A Turtle object representing the graphics turtle.
    """
    def draw_line_with_underscores(t, length=10):
        """Draws a line segment filled with underscores."""
        t.fillcolor("black")
        t.begin_fill()
        for _ in range(length // 3 + 1):  # Adjusting based on underscore count per unit
            t.down()
            t.write("_", align="center", font=("Arial", 8))
            t.up()
            t.forward(10)
        t.end_fill()

    def draw_pattern_level(t, level_number):
        """Draws one level of the pattern."""
        if level_number == 1:
            draw_line_with_underscores(t)
        else:
            t.penup()  # Move up before starting new level
            t.goto(-level_number * 10, 0)  # Position relative to previous levels
            t.pendown()
            draw_pattern_level(t, level_number - 1)
            t.penup()
            t.goto(level_number * 10, 0)
            t.pendown()
            draw_line_with_underscores(t)

    # Set initial position and orientation
    t.setheading(0)
    t.speed(0)  # Fastest speed for drawing lines quickly
    
    # Draw each level of the pattern
    for level_number in range(1, 5):
        print(f"Drawing Level {level_number}")
        draw_pattern_level(t, level_number)
        t.penup()
        t.goto(0, 0)  # Reset position after completing a level
'''
    code_list = [
        # code1,
        # code2,
        # code3,
        # code4,
        # code5,
        code6
    ]

    for idx, code in enumerate(code_list):
        filename = f"drawing_{idx}.png"
        image, result = Executor().run(
            code=code,
            show_screen=False,
            save_filename=filename,
            record_turtle_states=True
        )
        if result.get("status") == 'success':
            print(f"Generated image for drawing_{idx}: {filename}")
        else:
            print(f"Error in drawing_{idx}: {result['message']}")
