import re


def extract_draw_function(code):
    lines = code.splitlines()
    output_lines = []
    in_draw_function = False
    current_indent = None
    for line in lines:
        stripped_line = line.lstrip()
        if stripped_line.startswith('def draw('):
            in_draw_function = True
            current_indent = len(line) - len(stripped_line)
            output_lines.append(line)
        elif in_draw_function:
            indent = len(line) - len(line.lstrip())
            if indent > current_indent or line.strip() == '':
                output_lines.append(line)
            else:
                # End of draw function
                break
    return '\n'.join(output_lines)


def extract_python_code_from_text(text):
    """
    Extracts Python code from a given text block that contains the code within triple quotes. 
    If there are multiple code blocks, it returns all of them as a list. 
    If there are no code blocks, it returns an empty list.
    
    Args:
        text (str): The text containing code blocks
    
    Returns:
        list[str]: List of code blocks with whitespace stripped, or empty list if none found
    """
    def extract_matches(pattern):
        return re.findall(pattern, text, re.DOTALL)

    # Try to match content within triple quotes with language specified
    matches = extract_matches(r'```python([^`]*)```')
    if not matches:
        # Fallback to match content within generic triple quotes
        matches = extract_matches(r'```([^`]*)```')

    if matches:
        # Filter matches to include only those containing 'def draw('
        draw_matches = [match.strip() for match in matches if 'def draw(' in match]
        if draw_matches:
            return draw_matches
        # If no draw function found, return all matches
        return matches
    
    return []


def extract_json_code_from_text(text):
    """
    Extracts JSON code from a given text block that contains the code within triple quotes.
    """
    # Match content within triple quotes (```json ... ```)
    matches = re.findall(r'```json([^`]*)```', text, re.DOTALL)
    if matches:
        # Assuming the first match is the target code
        return matches[0].strip()
    return None