#!/bin/bash

export PYTHONPATH="./:${PYTHONPATH}"

source activate turtle

############################################################
# Logging
############################################################
log() {
    echo "$(tput setaf 3)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_done() {
    echo "$(tput setaf 2)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_error() {
    echo "$(tput setaf 1)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

############################################################
# Setup Xvfb
############################################################
if ! command -v Xvfb >/dev/null 2>&1; then
    log_error "Xvfb is not installed. Please install it first (e.g., 'apt-get install xvfb')"
    exit 1
fi

# use a random display number and try multiple times if needed
MAX_ATTEMPTS=10
DISPLAY_NUM=0
XVFB_STARTED=false

for attempt in $(seq 1 $MAX_ATTEMPTS); do
    DISPLAY_NUM=$((1000 + RANDOM % 1000))
    if Xvfb :${DISPLAY_NUM} -screen 0 1600x1200x16 &>/dev/null & then
        XVFB_PID=$!
        XVFB_STARTED=true
        break
    fi
    log "Display :${DISPLAY_NUM} is in use, trying another one..."
    sleep 1
done

if ! $XVFB_STARTED; then
    log_error "Failed to start Xvfb after ${MAX_ATTEMPTS} attempts. Please check if X server is already running or if there are permission issues."
    exit 1
fi

export DISPLAY=:${DISPLAY_NUM}.0
log "Successfully started Xvfb on display :${DISPLAY_NUM}"

# Add cleanup on script exit
trap "kill $XVFB_PID 2>/dev/null" EXIT

############################################################
# Check dependencies
############################################################
if ! command -v yq &> /dev/null; then
    log_error "yq is not installed. Please install it first."
    exit 1
fi

############################################################
# Evaluation
############################################################
evaluate_model() {
    local model_key=$1
    local config_file=$2
    
    # Get model configuration from YAML
    local model_family=$(yq -r ".[\"$model_key\"].model_family" "$config_file")
    local prompt_file=$(yq -r ".[\"$model_key\"].prompt_file" "$config_file")
    local response_file=$(yq -r ".[\"$model_key\"].response_file" "$config_file")
    local evaluation_file=$(yq -r ".[\"$model_key\"].evaluation_file" "$config_file")
    
    if [ -f "$evaluation_file" ]; then
        log_done "Evaluation file already exists for ${model_key}, skipping: ${evaluation_file}"
        return
    fi

    # Special handling for OpenAI responses
    if [[ "${model_key}" == "gpt-4o-2024-08-06" || "${model_key}" == "gpt-4-turbo-2024-04-09" || "${model_key}" == "gpt-4o-2024-11-20" ]]; then
        local jsonl_path="${response_file%.json}.jsonl"
        
        if [ -f "${jsonl_path}" ] && [ ! -f "${response_file}" ]; then
            log "Converting OpenAI JSONL responses to JSON format..."
            python src/llmutils/openai_jsonl_output_to_chat.py \
                --input_path "${jsonl_path}" \
                --output_path "${response_file}"
            log_done "Converted OpenAI JSONL responses to JSON format"
        fi
    fi

    log "Evaluating responses for ${model_key}..."

    if ! python src/turtlegfx/eval/eval_responses.py \
        --dataset_file "src/turtlegfx/data/graphics/dataset_graphics_sz823.json" \
        --prompt_file "${prompt_file}" \
        --response_file "${response_file}" \
        --output_file "${evaluation_file}" \
        --num_workers 8 \
        --use_embedding; then
        log_error "Failed to evaluate responses for ${model_key}"
    else
        log_done "Finished evaluation for ${model_key}"
    fi
}

############################################################
# Load and evaluate all models from YAML configs
############################################################
eval_all_models() {
    local config_dir="exps/eval_vlms/scripts/configs"
    
    # Using regex to match YAML files (similar to build_responses_vlms_submit_yaml.sh)
    local config_files=$(find "$config_dir" -regex ".*/.*\.yaml" ! -name "default_config.yaml")
    
    if [ -z "$config_files" ]; then
        log_error "No config files found in $config_dir"
        exit 1
    fi
    
    # Process each config file
    for config_file in $config_files; do
        log "Processing config file: $config_file"
        
        # Get all model keys from the current YAML file
        local model_keys=$(yq -r 'keys | .[]' "$config_file")
        
        # Evaluate all models in this config
        for model_key in $model_keys; do
            evaluate_model "$model_key" "$config_file"
        done
    done
}

############################################################
# Main
############################################################
eval_all_models 
# evaluate_model gpt-4o-2024-11-20 exps/eval_vlms/scripts/configs/base_models.yaml