import json
import argparse


def write_jsonl_file(requests, 
                     output_path, 
                     model="gpt-3.5-turbo-0125", 
                     max_tokens=1024, 
                     temperature=0, 
                     top_p=1):
    """
    Writes data to a .jsonl file formatted for OpenAI's Batch API.

    Args:
    - requests (list): List of request data where each element is a request with 'messages'.
    - output_path (str): Path to the output .jsonl file.
    - model (str): The model name to use for requests.
    - max_tokens (int): Maximum number of tokens for completion.
    - temperature (float): Controls randomness (0-2). Higher is more random.
    - top_p (float): Controls diversity via nucleus sampling (0-1).

    Returns:
    - None: Writes to the specified file.
    """
    assert isinstance(requests, list), "requests must be a list"
    assert all(isinstance(request, dict) and 'id' in request and 'message' in request for request in requests), "each request must be a dict with 'id' and 'message'"

    with open(output_path, 'w', encoding='utf-8') as f:
        for i, request in enumerate(requests):
            request_body = {
                "custom_id": request['id'],
                "method": "POST",
                "url": "/v1/chat/completions",
                "body": {
                    "model": model,
                    "messages": request['message'],
                    "max_tokens": max_tokens,
                    "temperature": temperature,
                    "top_p": top_p
                }
            }
            f.write(json.dumps(request_body, ensure_ascii=False) + '\n')
    print(f"JSONL file '{output_path}' created successfully.")

def main(input_path, output_path, model="gpt-3.5-turbo-0125", max_tokens=1024, temperature=0.7, top_p=1.0):
    """
    Main function to generate a JSONL file for OpenAI's Batch API.

    Args:
    - input_path (str): Path to the input JSON file with data.
    - output_path (str): Path to the output JSONL file.
    - model (str): Model name for the API requests.
    - max_tokens (int): Maximum number of tokens for completion.
    - temperature (float): Controls randomness (0-2). Higher is more random.
    - top_p (float): Controls diversity via nucleus sampling (0-1).
    """
    with open(input_path, 'r', encoding='utf-8') as f:
        data = json.load(f)

    write_jsonl_file(data, output_path, model=model, max_tokens=max_tokens, temperature=temperature, top_p=top_p)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Create a JSONL file for OpenAI Batch API.")
    parser.add_argument("--input_path", type=str, required=True, help="Path to the input JSON prompt file.")
    parser.add_argument("--output_path", type=str, required=True, help="Path to save the output JSONL file.")
    parser.add_argument("--model", type=str, default="gpt-4o", help="Model name to use for completions.")
    parser.add_argument("--max_tokens", type=int, default=1024, help="Maximum number of tokens for completions.")
    parser.add_argument("--temperature", type=float, default=0.7, help="Controls randomness (0-2). Higher is more random.")
    parser.add_argument("--top_p", type=float, default=1.0, help="Controls diversity via nucleus sampling (0-1).")

    args = parser.parse_args()
    main(args.input_path, args.output_path, model=args.model, max_tokens=args.max_tokens, temperature=args.temperature, top_p=args.top_p)
