#!/bin/bash

############################################################
# Environment Setup
############################################################
export PYTHONPATH="./:$PYTHONPATH"

############################################################
# Logging Functions
############################################################
log() {
    echo "$(tput setaf 0)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_warning() {
    echo "$(tput setaf 3)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_done() {
    echo "$(tput setaf 2)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_error() {
    echo "$(tput setaf 1)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

############################################################
# Check dependencies
############################################################
if ! command -v yq &> /dev/null; then
    log_error "Error: yq is not installed. Please install it first."
    exit 1
fi

############################################################
# Load configs
############################################################
CONFIG_DIR="exps/failure_analysis/scripts"
CONFIG_FILE="${CONFIG_DIR}/models_config.yaml"
PROMPT_CONFIG="${CONFIG_DIR}/prompt_config.yaml"

if [ ! -f "$CONFIG_FILE" ]; then
    log_error "Config file not found: $CONFIG_FILE"
    exit 1
fi

if [ ! -f "$PROMPT_CONFIG" ]; then
    log_error "Prompt config file not found: $PROMPT_CONFIG"
    exit 1
fi

############################################################
# Process each model
############################################################
process_model() {
    local model_key=$1
    
    # Get model configuration
    local dataset_file=$(yq -r ".[\"$model_key\"].dataset_file" "$CONFIG_FILE")
    local prompt_file=$(yq -r ".[\"$model_key\"].prompt_file" "$CONFIG_FILE")
    
    # Get prompt configuration
    local prompt_template=$(yq -r ".prompt_template" "$PROMPT_CONFIG")
    local build_prompts_script=$(yq -r ".build_prompts_script" "$PROMPT_CONFIG")
    local python=$(yq -r ".python" "$PROMPT_CONFIG")
    
    log "Processing model: $model_key"
    log "Dataset file: $dataset_file"
    log "Prompt file: $prompt_file"
    
    # Check if output file already exists
    if [ -f "$prompt_file" ]; then
        log_warning "Skipping $model_key (prompt file already exists)"
        return
    fi
    
    # Build prompts
    if ! ${python} ${build_prompts_script} \
        --input_file ${dataset_file} \
        --prompt_template ${prompt_template} \
        --output_file ${prompt_file}; then
        log_error "Failed to build prompts for ${model_key}"
        return
    fi
    
    log_done "Successfully built prompts for ${model_key}"
}

############################################################
# Main execution
############################################################
# Get all model keys from the config file
model_keys=$(yq -r 'keys | .[]' "$CONFIG_FILE")

# Process each model
for model_key in $model_keys; do
    process_model "$model_key"
done

log_done "All prompts building complete"
