import pytest
from src.turtlegfx.emulate.emulator import Executor, ERROR_NO_CODE_PROVIDED, ERROR_NO_FUNC_FOUND

class TestExecutor:
    def setup_method(self):
        self.executor = Executor()
        
        # Sample valid code that draws a square
        self.valid_code = '''
def draw(t):
    for _ in range(4):
        t.forward(100)
        t.right(90)
'''

        self.valid_code_with_fill = '''
def draw(t):
    for _ in range(4):
        t.forward(100)
        t.right(90)
    t.fillcolor("red")
    t.begin_fill()
    t.circle(50)
    t.end_fill()
'''
        
        # Code without a draw function
        self.no_draw_func_code = '''
def something_else(t):
    pass
'''
        
        # Code with syntax error
        self.invalid_syntax_code = '''
def draw(t):
    for _ in range(4)
        t.forward(100)  # Missing colon after range(4)
'''

    def test_valid_code_execution(self):
        """Test execution of valid turtle code"""
        image, result = self.executor.run(code=self.valid_code, show_screen=False)
        
        assert result["status"] == "success"
        assert result["error_type"] == "EXEC_SUCCESS"
        assert image is not None

    def test_no_code_provided(self):
        """Test handling when no code is provided"""
        image, result = self.executor.run(code=None, show_screen=False)
        
        assert result["status"] == "fail"
        assert result["error_type"] == ERROR_NO_CODE_PROVIDED
        assert image is None

    def test_no_draw_function(self):
        """Test handling when code doesn't contain a draw function"""
        image, result = self.executor.run(code=self.no_draw_func_code, show_screen=False)
        
        assert result["status"] == "fail"
        assert result["error_type"] == ERROR_NO_FUNC_FOUND
        assert image is None

    def test_invalid_syntax(self):
        """Test handling of code with syntax errors"""
        image, result = self.executor.run(code=self.invalid_syntax_code, show_screen=False)
        
        assert result["status"] == "fail"
        assert image is None

    def test_model_output_execution(self):
        """Test execution of code extracted from model output"""
        model_output = """Here's the code to draw a triangle:\n```python\ndef draw(t):
            for _ in range(3):
                t.forward(100)
                t.right(120)
            ```
        """
        
        image, result = self.executor.run_model_output(
            model_output=model_output,
            show_screen=False
        )
        
        assert result["status"] == "success"
        assert result["error_type"] == "EXEC_SUCCESS"
        assert image is not None

    def test_save_to_file(self, tmp_path):
        """Test saving the output to a file"""
        save_path = tmp_path / "test_drawing.png"
        
        image, result = self.executor.run(
            code=self.valid_code,
            show_screen=False,
            save_filename=str(save_path)
        )
        
        assert result["status"] == "success"
        assert save_path.exists()
        assert save_path.stat().st_size > 0

    def test_turtle_state_recording(self):
        """Test recording of turtle states"""

        
        image, result = self.executor.run(
            code=self.valid_code,
            show_screen=False,
            record_turtle_states=True
        )

        assert result["status"] == "success"
        assert "turtle_states" in result
        assert isinstance(result["turtle_states"], dict)
        assert 'states' in result["turtle_states"]
        assert len(result["turtle_states"]["states"]) > 0

        image_fill, result_fill = self.executor.run(
            code=self.valid_code_with_fill,
            show_screen=False,
            record_turtle_states=True
        )
        
        assert result_fill["status"] == "success"
        assert "turtle_states" in result_fill
        assert isinstance(result_fill["turtle_states"], dict)
        assert 'states' in result_fill["turtle_states"]
        assert len(result_fill["turtle_states"]["states"]) > 0

        for state in result_fill["turtle_states"]["states"]:
            print(str(state))

if __name__ == "__main__":
        # To see the output of these tests, run any of these commands:
        # pytest -s src/tests/test_emulator.py          # Shows all print output
        # pytest -v src/tests/test_emulator.py          # Shows verbose test results
        # pytest -sv src/tests/test_emulator.py         # Shows both verbose and print output
        # pytest -sv src/tests/test_emulator.py::TestExecutor::test_turtle_state_recording  # Run only this test
    pytest.main([__file__]) 