import unittest
from src.turtlegfx.eval.lines import compare_lines_from_code
import logging

logging.basicConfig(level=logging.DEBUG)


class TestCompareLines(unittest.TestCase):
    code_midi_11j_chatcmpl_f34a22a4_output = '''def draw(t):
    """Draw a green star pattern."""
    import turtle

    # Set up the turtle
    t.speed(0)
    t.pensize(2)
    t.color("black", "green")

    # Function to draw a single diamond shape
    def draw_diamond():
        t.begin_fill()
        for _ in range(2):
            t.forward(100)
            t.right(45)
            t.forward(100)
            t.right(135)
        t.end_fill()

    # Draw the star pattern
    for _ in range(8):
        draw_diamond()
        t.right(45)

    # Hide the turtle after drawing
    t.hideturtle()
    '''

    code_midi_11j_chatcmpl_f34a22a4_true = '''def draw(t):
    # Set the fill color to green
    t.fillcolor('green')

    # Begin filling the shape
    t.begin_fill()

    # Define a nested function to draw an octagonal star pattern
    def draw_octagonal_star():
        for _ in range(8):
            for _ in range(2):
                t.forward(100)
                t.right(45)
                t.forward(100)
                t.right(135)
            t.right(45)

    # Draw the octagonal star pattern
    draw_octagonal_star()

    # End filling the shape
    t.end_fill()
    '''
    
    code_midi_10h_chatcmpl_32cde675_output = '''def draw(t):
    import turtle

    # Function to draw a filled circle
    def draw_circle(t, color):
        t.fillcolor(color)
        t.begin_fill()
        t.circle(50)
        t.end_fill()

    # Set the speed and pen size
    t.speed(0)
    t.pensize(1)

    # Define the colors for the circles
    colors = [
        ["blue", "green", "red"],
        ["yellow", "blue", "green"],
        ["red", "yellow", "blue"]
    ]

    # Starting position
    start_x = -100
    start_y = 100

    # Draw the grid of circles
    for row in range(3):
        for col in range(3):
            t.penup()
            t.goto(start_x + col * 100, start_y - row * 100)
            t.pendown()
            draw_circle(t, colors[row][col])

    # Hide the turtle
    t.hideturtle()
    '''

    code_midi_10h_chatcmpl_32cde675_true = '''def draw(t):
    t.setheading(90)

    # Function to draw a circle with a given radius and color
    def draw_circle(radius, color):
        t.fillcolor(color)
        t.begin_fill()
        t.circle(radius)
        t.end_fill()

    # Define a list of colors for the pattern
    colors = ['red', 'yellow', 'blue', 'green']

    # Draw a pattern of alternating colored circles
    for i in range(3):
        for j in range(3):
            t.penup()
            t.goto(j * 150 - 225, i * 150 - 225)
            t.pendown()
            draw_circle(75, colors[(i + j) % len(colors)])
    '''

    code_midi_8g_chatcmpl_f466813c_output = '''def draw(t):
    """Draw a circle."""
    t.penup()
    t.goto(0, -50)  # Move to starting position
    t.pendown()
    t.circle(50)  # Draw a circle with radius 50
    '''

    code_midi_8g_chatcmpl_f466813c_true = '''def draw(t):
    t.setheading(0)  # Change initial direction to 0 degrees

    for _ in range(6):  # Maintain the hexagon shape
        t.circle(25)  # Draw a circle instead of a zigzag pattern
    '''
    
    code_midi_8d_chatcmpl_35e23aa7_output = '''def draw(t):
    """Draw a geometric pattern of overlapping circles."""
    import math

    def draw_circle(t, radius):
        """Draw a circle with the given radius."""
        circumference = 2 * math.pi * radius
        step_length = circumference / 360
        step_angle = 1

        for _ in range(360):
            t.forward(step_length)
            t.left(step_angle)

    # Set the initial position and angle
    t.penup()
    t.goto(0, -100)  # Start at the bottom of the first circle
    t.pendown()

    # Draw the pattern
    num_circles = 12
    angle_between_circles = 360 / num_circles
    radius = 100

    for _ in range(num_circles):
        draw_circle(t, radius)
        t.left(angle_between_circles)
    '''

    code_midi_8d_chatcmpl_35e23aa7_true = '''def draw(t):
    t.setheading(90)

    for _ in range(12):
        t.circle(100)
        t.right(30)
    '''
    
    code_midi_7b_chatcmpl_4bc4a92b_output = '''def draw(t):
    """Draw the specified geometric pattern."""
    import turtle

    def draw_square(t, size, color):
        t.fillcolor(color)
        t.begin_fill()
        for _ in range(4):
            t.forward(size)
            t.right(90)
        t.end_fill()

    def draw_pattern(t, x, y, outer_color, inner_color):
        t.penup()
        t.goto(x, y)
        t.pendown()
        draw_square(t, 100, outer_color)
        t.penup()
        t.goto(x + 25, y - 25)
        t.pendown()
        draw_square(t, 50, inner_color)

    # Set up the turtle
    t.speed(0)
    t.penup()
    t.goto(-150, 50)
    t.pendown()

    # Draw the pattern
    colors = [("blue", "red"), ("red", "blue")]
    for row in range(2):
        for col in range(3):
            outer_color, inner_color = colors[(row + col) % 2]
            draw_pattern(t, -150 + col * 100, 50 - row * 100, outer_color, inner_color)

    # Hide the turtle
    t.hideturtle()
    '''

    code_midi_7b_chatcmpl_4bc4a92b_true = '''def draw(t):
    t.setheading(90)

    def draw_rectangle(color, x, y, width, height):
        t.penup()
        t.goto(x, y)
        t.pendown()
        t.fillcolor(color)
        t.begin_fill()
        for _ in range(2):
            t.forward(width)
            t.right(90)
            t.forward(height)
            t.right(90)
        t.end_fill()

    def draw_alternating_rectangles(color1, color2, x, y, width, height):
        draw_rectangle(color1, x, y, width, height)
        if (x // width) % 2 == 0:  # Alternate positions
            draw_rectangle(color2, x + width // 4, y + height // 4, width // 2, height // 2)
        else:
            draw_rectangle(color2, x + width // 2 - width // 4, y + height // 4, width // 2, height // 2)

    colors = ["blue", "red"]
    width = 75  # Width of each rectangle
    height = 75  # Height of each rectangle
    start_x = 0  # Adjust starting x-coordinate
    start_y = 0  # Adjust starting y-coordinate

    for row in range(2):  # Two rows
        for col in range(3):  # Three columns
            color = colors[(row + col) % 2]  # Alternate colors
            x = start_x + col * width
            y = start_y - row * height
            draw_alternating_rectangles(color, colors[(row + col + 1) % 2], x, y, width, height)
    '''

    code_midi_7d_chatcmpl_f8f375e5_output = '''def draw(t):
    """Draw a 2x4 grid of squares with the top-left square outlined in red."""
    t.penup()
    t.goto(-100, 50)  # Start at the top-left corner
    t.pendown()

    def draw_square(t, size, outline_color="black"):
        t.pencolor(outline_color)
        for _ in range(4):
            t.forward(size)
            t.right(90)

    square_size = 50

    # Draw the top row
    for i in range(4):
        if i == 0:
            draw_square(t, square_size, "red")
        else:
            draw_square(t, square_size)
        t.forward(square_size)

    # Move to the start of the bottom row
    t.penup()
    t.goto(-100, 50 - square_size)
    t.pendown()

    # Draw the bottom row
    for _ in range(4):
        draw_square(t, square_size)
        t.forward(square_size)
    '''

    code_midi_7d_chatcmpl_f8f375e5_true = '''def draw(t):
    t.setheading(90)

    # Define a helper function to draw a square
    def draw_square(t, side_length):
        """Draw a square with a given side length."""
        for _ in range(4):
            t.forward(side_length)
            t.right(90)

    # Define a helper function to draw rotated squares
    def draw_rotated_squares(t, side_length, num_squares):
        """Draw squares rotated around a central point."""
        for _ in range(num_squares):
            draw_square(t, side_length)
            t.right(360 / num_squares)  # Rotate for the next square

    # Draw a series of rotated squares with color changes
    t.pencolor('black')
    for _ in range(3):
        draw_rotated_squares(t, 50, 4)  # Draw 4 squares rotated around a central point
        t.pencolor('white')
        t.forward(50)
        t.pencolor('black')
        t.backward(50)
        t.left(90)
        t.forward(50)
        t.right(90)

    # Draw a final square with a different color
    t.pencolor('red')
    draw_square(t, 50)
    '''

    
    code_midi_3j_chatcmpl_c2c70e38_output = '''def draw(t):
    # Draw the large blue square
    t.penup()
    t.goto(-200, -200)  # Move to the bottom-left corner of the square
    t.pendown()
    t.fillcolor("blue")
    t.begin_fill()
    for _ in range(4):
        t.forward(400)
        t.left(90)
    t.end_fill()

    # Draw the small white triangle
    t.penup()
    t.goto(-200, -200)  # Move to the bottom-left corner of the triangle
    t.pendown()
    t.fillcolor("white")
    t.begin_fill()
    t.forward(100)
    t.left(135)
    t.forward(141.42)  # Hypotenuse of the triangle (100 * sqrt(2))
    t.left(135)
    t.forward(100)
    t.end_fill()

    # Hide the turtle
    t.hideturtle()
    '''

    code_midi_3j_chatcmpl_c2c70e38_true = '''def draw(t):
    # Set the initial heading to face upwards
    t.setheading(90)

    # Define the number of repetitions for the pattern
    repetitions = 1  # Reduced repetitions for a simpler shape

    # Define the step size for the pattern
    step_size = 150  # Increased step size for a larger shape

    # Define the number of squares in each row
    squares_per_row = 1  # Reduced squares per row for a simpler shape

    # Define the fill color
    t.fillcolor('blue')  # Simplified color scheme

    # Start filling the shape
    t.begin_fill()

    # Draw a larger square
    for _ in range(4):
        t.forward(step_size * 3)  # Increased length for a larger shape
        t.right(90)

    # Extend the shape to the right
    t.forward(step_size)
    t.right(90)
    t.forward(step_size)
    t.left(90)

    # End filling the shape
    t.end_fill()
    '''
    
    code_midi_3j_chatcmpl_aa8e315c_output = '''def draw(t):
    """Draw a square with a diagonal split into two triangles, one black and one white."""
    # Set up the turtle
    t.speed(0)
    t.penup()
    t.goto(-50, 50)  # Start at the top-left corner of the square
    t.pendown()

    # Draw the square
    t.color("black")
    t.forward(100)  # Top side
    t.right(90)
    t.forward(100)  # Right side
    t.right(90)
    t.forward(100)  # Bottom side
    t.right(90)
    t.forward(100)  # Left side
    t.right(90)

    # Draw the diagonal line
    t.penup()
    t.goto(-50, 50)
    t.pendown()
    t.goto(50, -50)

    # Fill the bottom-right triangle with black
    t.penup()
    t.goto(50, -50)
    t.pendown()
    t.begin_fill()
    t.goto(50, 50)
    t.goto(-50, -50)
    t.goto(50, -50)
    t.end_fill()
    '''

    code_midi_3j_chatcmpl_aa8e315c_true = '''def draw(t):
    # Set the initial heading to face upwards
    t.setheading(90)
    
    # Define a function to draw a line of a given length and direction
    def draw_line(length, direction):
        """Draw a line of a given length."""
        t.forward(length)
        if direction == 'right':
            t.right(90)
        elif direction == 'left':
            t.left(90)
    
    # Define a function to draw multiple connected lines
    def draw_connected_lines(length, num_lines, direction):
        for _ in range(num_lines):
            draw_line(length, direction)
    
    # Set the fill color to black
    t.fillcolor('black')
    
    # Start the fill operation
    t.begin_fill()
    
    # Draw a hexagon with connected lines
    draw_connected_lines(50, 6, 'right')
    
    # End the fill operation
    t.end_fill()
    '''
    
    code_midi_11j_chatcmpl_cb7f1d90_output = '''def draw(t):
    """Draw the specified geometric pattern."""
    import turtle

    def draw_square(t, size, color):
        t.fillcolor(color)
        t.begin_fill()
        for _ in range(4):
            t.forward(size)
            t.right(90)
        t.end_fill()

    def draw_pattern(t, size):
        # Draw the outer blue square
        t.penup()
        t.goto(-size / 2, size / 2)
        t.pendown()
        draw_square(t, size, 'blue')

        # Draw the middle blue square
        t.penup()
        t.goto(-size / 4, size / 4)
        t.pendown()
        draw_square(t, size / 2, 'blue')

        # Draw the inner white square
        t.penup()
        t.goto(-size / 8, size / 8)
        t.pendown()
        draw_square(t, size / 4, 'white')

        # Draw the black lines
        t.penup()
        t.goto(-size / 2, 0)
        t.pendown()
        t.goto(size / 2, 0)

        t.penup()
        t.goto(0, size / 2)
        t.pendown()
        t.goto(0, -size / 2)

    # Set up the turtle
    t.speed(0)
    t.pensize(1)
    t.color('black')

    # Draw the pattern
    draw_pattern(t, 200)
    '''

    code_midi_11j_chatcmpl_cb7f1d90_true = '''def draw(t):
    import math

    def draw_diamond(t, side_length, num_sides):
        for _ in range(num_sides):
            t.forward(side_length)
            t.right(90)
            t.forward(side_length)
            t.right(90)

    def draw_diamond_pattern(t, side_length, num_sides, num_repeats):
        for _ in range(num_repeats):
            draw_diamond(t, side_length, num_sides)
            t.right(360 / num_repeats)

    # Fill the area between two diamond patterns with blue
    t.fillcolor('blue')
    t.begin_fill()

    # Draw the central square pattern
    draw_diamond_pattern(t, 100, 2, 4)

    # Draw an extended diamond pattern with reduced complexity
    for i in range(2):
        draw_diamond_pattern(t, 100 + i * 50, 2, 4)
        t.left(90)

    # End filling
    t.end_fill()

    # Draw a smaller diamond pattern in the center
    t.penup()
    t.goto(0, 0)
    t.pendown()
    t.color('black', 'white')  # Set fill color to white to overwrite blue fill
    t.begin_fill()
    draw_diamond_pattern(t, 50, 2, 4)
    t.end_fill()
    '''

    code_red_square_white_square = '''def draw(t):
    """Draw a red square with a smaller white square inside."""
    # Set up the turtle
    t.speed(0)
    
    # Draw the large red square
    t.penup()
    t.goto(-100, -100)  # Start at bottom-left corner of the large square
    t.pendown()
    t.fillcolor("red")
    t.begin_fill()
    for _ in range(4):
        t.forward(200)
        t.left(90)
    t.end_fill()
    
    # Draw the smaller white square
    t.penup()
    t.goto(-100, 0)  # Start at bottom-left corner of the smaller square
    t.pendown()
    t.fillcolor("white")
    t.begin_fill()
    for _ in range(4):
        t.forward(100)
        t.left(90)
    t.end_fill()
    '''
    
    code_red_square_white_square_2 = '''def draw(t):
    t.setheading(90)

    # Alternating color pattern
    colors = ['red', 'blue']
    t.fillcolor(colors[0])  # Fill with the first color
    t.pencolor(colors[0])   # Set the pen color to the first color
    t.begin_fill()
    
    # Draw a larger square
    for _ in range(4):
        t.forward(200)  # Increase the length to create a larger shape
        t.right(90)
    
    # Extend the shape to the right
    t.forward(100)
    t.right(90)
    t.forward(100)
    t.left(90)
    
    # Add an extension to the bottom
    t.forward(100)
    t.left(90)
    t.forward(100)
    t.right(90)
    
    t.end_fill()
    '''
    sparse_radio_1 = '''def draw(t):
    """Draw a starburst pattern with 12 lines."""
    import turtle

    t.color("purple")
    t.pensize(2)

    for _ in range(12):
        t.forward(100)
        t.backward(100)
        t.right(30)
    '''

    sparse_radio_2 = '''def draw(t):
    # Draw a complex pattern with color and rotation
    t.setheading(90)
    for _ in range(12):
        t.pencolor('green')
        t.forward(75)
        t.backward(25)
        t.pencolor('purple')
        t.forward(25)
        t.backward(75)
        t.right(30)
    '''
    radio_1 = '''def draw(t):
    """Draw a radial pattern with lines."""
    import turtle

    t.speed(0)  # Set the fastest drawing speed
    t.color("blue")  # Set the color to blue

    num_lines = 36  # Number of lines in the pattern
    angle = 360 / num_lines  # Angle between each line

    for _ in range(num_lines):
        t.penup()
        t.goto(0, 0)  # Start from the center
        t.pendown()
        t.forward(100)  # Draw a line outward
        t.penup()
        t.goto(0, 0)  # Return to the center
        t.right(angle)  # Rotate to the next angle
    '''
    
    radio_2 = '''def draw(t):
    # Define a reusable function to draw a colored polygon
    def draw_colored_polygon(size, sides, color1, color2):
        t.pencolor(color1)
        for _ in range(sides):
            t.forward(size)
            t.pencolor(color2)
            t.forward(size // 2)
            t.pencolor(color1)
            t.backward(size * 1.5)
            t.right(360 / sides)

    # Define a reusable function to draw a pattern of polygons
    def draw_colored_pattern(size, sides, color1, color2, repetitions):
        for _ in range(repetitions):
            draw_colored_polygon(size, sides, color1, color2)
            t.right(360 / repetitions)

    # Draw a pattern of nonagons with alternating colors
    draw_colored_pattern(100, 9, 'blue', 'red', 4)
    '''

    cross_11j_code_1 = '''def draw(t):
    """Draw a red cross pattern."""
    t.color("red")
    t.pensize(2)

    # Draw vertical line
    t.penup()
    t.goto(0, -50)
    t.pendown()
    t.goto(0, 50)

    # Draw horizontal line
    t.penup()
    t.goto(-50, 0)
    t.pendown()
    t.goto(50, 0)
    '''
    
    cross_11j_code_2 = '''def draw(t):
    t.setheading(0)  # Change the initial heading to 0 degrees

    # Fill color
    t.fillcolor('black')
    t.begin_fill()
    for _ in range(4):  # Increase the number of iterations
        t.pencolor('red')  # Change the pen color to red
        t.forward(75)  # Move forward by 75 units
        t.pencolor('yellow')  # Change the pen color to yellow
        t.forward(25)  # Move forward by 25 units
        t.pencolor('red')  # Change the pen color back to red
        t.backward(100)  # Move backward by 100 units

        t.right(90)  # Turn right by 90 degrees
    t.end_fill()
    '''
    
    octagon_color_code_1 = '''def draw(t):
    """Draw an octagon with alternating blue and green sides."""
    t.penup()
    t.goto(-50, 0)  # Start position
    t.pendown()
    
    side_length = 100
    angle = 45  # Internal angle for an octagon

    for i in range(8):
        if i % 2 == 0:
            t.pencolor("blue")
        else:
            t.pencolor("green")
        t.forward(side_length)
        t.left(angle)
    '''
    octagon_color_code_2 = '''def draw(t):
    t.setheading(90)

    # draw an octagon with alternating colors
    colors = ['blue', 'green']
    for i in range(8):
        t.pencolor(colors[i % 2])
        t.forward(100)
        t.right(45)
    '''
    octagon_code_1 = '''
def draw(t):
    """
    Draws an octagon centered on the origin.
    
    Parameters:
    t (Turtle): A pre-created Turtle instance used for drawing.
    """
    # Set initial position and orientation
    t.penup()
    t.goto(0, 0)
    t.pendown()

    # Define the size of each side of the octagon
    side_length = 50
    
    # Loop through each vertex of the octagon
    for _ in range(8):
        # Move forward by the side length
        t.forward(side_length)
        
        # Turn left by 45 degrees
        t.left(45)
'''
    octagon_code_2 = '''
def draw(t):
    for _ in range(8):
        t.forward(100)
        t.right(45)
'''

    llava_output_code = '''
def draw(t):
    """
    Draws a square frame around the text "NEC DEYFES" without filling it.
    
    Parameters:
    t (turtle.Turtle): A pre-created turtle instance used for drawing.
    """
    # Set pen color to black for outline only
    t.color("black")
    
    # Move forward by half the width of the text box
    t.penup()
    t.goto(-75, 0)
    t.pendown()
    
    # Draw top edge of the text box
    t.forward(150)
    
    # Turn left to face the left side of the text box
    t.left(90)
    
    # Draw left edge of the text box
    t.forward(150)
    
    # Turn left again to face the bottom edge
    t.left(90)
    
    # Draw bottom edge of the text box
    t.forward(150)
    
    # Finally turn left to complete the loop
    t.left(90)
    
    # Lift the pen up after completing the drawing
    t.penup()
'''
    method_11g_code_1 = """
def draw(t):
    t.setheading(60)
    
    def draw_diamond(side_length):
        for _ in range(2):
            t.forward(side_length)
            t.right(60)
            t.forward(side_length)
            t.right(120)
    
    def draw_triangle(side_length):
        for _ in range(3):
            t.forward(side_length)
            t.right(120)
    
    # Set the side length
    side_length = 100

    for _ in range(6):
        draw_diamond(side_length)
        t.right(60)

    t.setheading(0)
    for _ in range(6):
        draw_triangle(side_length)
        t.right(60)
"""

    method_11g_code_2 = """
def draw(t):
    import math
    t.setheading(60)
    
    def draw_diamond(side_length):
        for _ in range(2):
            t.forward(side_length)
            t.right(60)
            t.forward(side_length)
            t.right(120)
    
    def draw_hexagon(side_length):
        for _ in range(6):
            t.forward(side_length)
            t.right(60)
    
    # Set the side length
    side_length = 100

    for _ in range(6):
        draw_diamond(side_length)
        t.right(60)

    t.penup()
    t.goto(50, 50*math.sqrt(3))
    t.pendown()

    t.setheading(-60)
    draw_hexagon(side_length)
"""
    line_200_one_step_code = """def draw(t):
            t.forward(200)
        """
    
    line_200_two_step_code = """def draw(t):
            t.forward(100)
            t.forward(100)
        """
    
    line_100_100_steps_code = """def draw(t):
            # move to somewhere else
            t.penup()
            t.goto(30, 123)
            t.pendown()
            for _ in range(100):
                t.forward(1)
        """
    
    square_200_one_step_code = """def draw(t):
            for _ in range(4):
                t.forward(200)
                t.right(90)
        """
        
    square_200_two_step_code = """def draw(t):
            for _ in range(4):
                t.forward(100)
                t.forward(100)
                t.left(90)
        """
    
    # Existing test code definitions
    square_100_code = """def draw(t):
            for _ in range(4):
                t.forward(100)
                t.left(90)
        """
    
    square_200_code = """def draw(t):
            for _ in range(4):
                t.forward(200)
                t.right(90)
        """
    
    square_nested_code = """def draw(t):
            def draw_square(t):
                for _ in range(4):
                    t.forward(100)
                    t.left(90)
            draw_square(t)
        """
    
    triangle_100_code = """def draw(t):
            for _ in range(3):
                t.forward(100)
                t.left(120)
        """
    
    triangle_200_code = """def draw(t):
            for _ in range(3):
                t.forward(200)
                t.left(120)
        """
    
    square_offset_code = """def draw(t):
            t.penup()
            t.goto(50, 50)
            t.pendown()
            for _ in range(4):
                t.forward(100)
                t.left(90)
        """
    
    lines_ordered_code = """def draw(t):
            for _ in range(4):
                t.forward(100)
                t.back(100)
                t.right(90)
        """
    
    lines_unordered_code = """def draw(t):
            t.forward(100)
            t.back(100)
            t.back(100)
            t.forward(100)
            t.right(90)
            t.forward(100)
            t.back(100)
            t.back(100)
        """
    
    filled_square_red_code = """def draw(t):
            t.fillcolor('red')
            t.begin_fill()
            for _ in range(4):
                t.forward(100)
                t.left(90)
            t.end_fill()
        """
    
    filled_square_blue_code = """def draw(t):
            t.fillcolor('blue')
            t.begin_fill()
            for _ in range(4):
                t.forward(100)
                t.left(90)
            t.end_fill()
        """
    
    filled_triangle_code = """def draw(t):
            t.begin_fill()
            for _ in range(3):
                t.forward(100)
                t.left(120)
            t.end_fill()
        """
    
    filled_square_outline_code = """def draw(t):
            t.fillcolor('red')
            t.begin_fill()
            for _ in range(4):
                t.forward(100)
                t.left(90)
            t.end_fill()
            t.forward(100)
        """
    
    # Additional code snippets
    # 1. Different Pen Sizes
    square_pensize_1_code = """def draw(t):
        t.pensize(1)
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    square_pensize_5_code = """def draw(t):
        t.pensize(5)
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    # 2. Different Pen Colors
    square_pencolor_red_code = """def draw(t):
        t.pencolor('red')
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    square_pencolor_blue_code = """def draw(t):
        t.pencolor('blue')
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    # 3. Empty Code
    empty_code = """def draw(t):
        pass
    """
    
    # 4. Discontinuous Lines (Pen Up and Pen Down)
    discontinuous_line_code = """def draw(t):
        t.forward(50)
        t.penup()
        t.forward(50)
        t.pendown()
        t.forward(50)
    """
    
    continuous_line_code = """def draw(t):
        t.forward(150)
    """
    
    # 5. Rotated Shapes
    square_rotated_45_code = """def draw(t):
        t.left(45)
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    # 6. Negative Coordinates
    square_negative_coords_code = """def draw(t):
        t.penup()
        t.goto(-50, -50)
        t.pendown()
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    # 7. Complex Shapes (Star)
    star_code = """def draw(t):
        for _ in range(5):
            t.forward(100)
            t.right(144)
    """
    
    star_alternative_code = """def draw(t):
        for _ in range(5):
            t.forward(100)
            t.right(144)
    """
    
    # 8. Circles and Arcs
    circle_100_code = """def draw(t):
        t.circle(100)
    """
    
    circle_100_alternative_code = """def draw(t):
        t.circle(100)
    """
    
    # 9. Filled Shapes with Different Start Positions
    filled_square_offset_code = """def draw(t):
        t.penup()
        t.goto(100, 100)
        t.pendown()
        t.fillcolor('red')
        t.begin_fill()
        for _ in range(4):
            t.forward(100)
            t.left(90)
        t.end_fill()
    """
    
    # 10. Shapes with Different Drawing Orders (Overlapping Shapes)
    overlapping_shapes_1_code = """def draw(t):
        t.fillcolor('blue')
        t.begin_fill()
        for _ in range(4):
            t.forward(100)
            t.left(90)
        t.end_fill()
        t.fillcolor('red')
        t.begin_fill()
        for _ in range(3):
            t.forward(100)
            t.left(120)
        t.end_fill()
    """
    
    overlapping_shapes_2_code = """def draw(t):
        t.fillcolor('red')
        t.begin_fill()
        for _ in range(3):
            t.forward(100)
            t.left(120)
        t.end_fill()
        t.fillcolor('blue')
        t.begin_fill()
        for _ in range(4):
            t.forward(100)
            t.left(90)
        t.end_fill()
    """
    
    # 11. Testing Code with Errors (Syntax Errors)
    code_with_error_code = """def draw(t):
        for _ in range(4)
            t.forward(100)
            t.left(90)
    """
    
    # 12. Shapes with Stamps
    code_with_stamp_code = """def draw(t):
        t.shape('turtle')
        t.stamp()
    """
    
    # 13. Speed Variations
    square_speed_1_code = """def draw(t):
        t.speed(1)
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    square_speed_10_code = """def draw(t):
        t.speed(10)
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    # 15. Pen Up/Down Affecting Line Continuity
    complex_penup_pendown_1_code = """def draw(t):
        t.penup()
        t.goto(0, 0)
        t.pendown()
        t.forward(100)
        t.penup()
        t.forward(50)
        t.pendown()
        t.forward(100)
    """
    
    complex_penup_pendown_2_code = """def draw(t):
        t.penup()
        t.goto(0, 0)
        t.pendown()
        t.forward(250)
    """
    
    # 16. Relative vs Absolute Movements
    relative_movement_code = """def draw(t):
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    absolute_movement_code = """def draw(t):
        t.goto(0, 0)
        t.goto(100, 0)
        t.goto(100, 100)
        t.goto(0, 100)
        t.goto(0, 0)
    """
    
    # 17. Mirrored Shapes
    square_mirrored_code = """def draw(t):
        t.forward(100)
        t.right(90)
        t.forward(100)
        t.right(90)
        t.forward(100)
        t.right(90)
        t.forward(100)
    """
    
    # 18. Shapes Drawn in Opposite Directions
    square_counter_clockwise_code = """def draw(t):
        for _ in range(4):
            t.forward(100)
            t.left(90)
    """
    
    square_clockwise_code = """def draw(t):
        for _ in range(4):
            t.forward(100)
            t.right(90)
    """
    
    def test_line_length_invariant(self):
        self.assertTrue(compare_lines_from_code(self.line_200_one_step_code, self.line_200_two_step_code, show_screen=False, length_invariant=True))
        self.assertTrue(compare_lines_from_code(self.line_200_one_step_code, self.line_100_100_steps_code, show_screen=False, length_invariant=True))
    
    # Existing test methods
    def test_identical_squares(self):
        self.assertTrue(compare_lines_from_code(self.square_100_code, self.square_nested_code, show_screen=False))
    
    def test_different_shapes(self):
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.triangle_100_code, show_screen=False))
    
    def test_same_shape_different_size(self):
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.square_200_code, show_screen=False, length_invariant=False, position_invariant=True))
        self.assertTrue(compare_lines_from_code(self.square_100_code, self.square_200_code, show_screen=False, length_invariant=True, position_invariant=True))
        self.assertFalse(compare_lines_from_code(self.triangle_100_code, self.triangle_200_code, show_screen=False, length_invariant=False, position_invariant=True))
        self.assertTrue(compare_lines_from_code(self.triangle_100_code, self.triangle_200_code, show_screen=False, length_invariant=True, position_invariant=True))
    
    def test_same_shape_different_position(self):
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.square_offset_code, show_screen=False, length_invariant=True, position_invariant=False))
        self.assertTrue(compare_lines_from_code(self.square_100_code, self.square_offset_code, show_screen=False, length_invariant=True, position_invariant=True))
    
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.square_200_code, show_screen=False, length_invariant=True, position_invariant=False))
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.square_200_code, show_screen=False, length_invariant=False, position_invariant=True))
        self.assertTrue(compare_lines_from_code(self.square_100_code, self.square_200_code, show_screen=False, length_invariant=True, position_invariant=True))
    
    def test_same_lines_different_order(self):
        self.assertTrue(compare_lines_from_code(self.lines_ordered_code, self.lines_unordered_code, show_screen=False))
    
    def test_identical_filled_shapes(self):
        self.assertTrue(compare_lines_from_code(self.filled_square_red_code, self.filled_square_red_code, show_screen=False))
    
    def test_different_filled_shapes(self):
        self.assertFalse(compare_lines_from_code(self.filled_square_red_code, self.filled_triangle_code, show_screen=False))
    
    def test_same_shape_different_fillcolor(self):
        self.assertFalse(compare_lines_from_code(self.filled_square_red_code, self.filled_square_blue_code, show_screen=False))
    
    def test_filled_shape_with_outline(self):
        self.assertTrue(compare_lines_from_code(self.filled_square_red_code, self.filled_square_outline_code, show_screen=False))
    
    # Additional test methods
    def test_same_shape_different_pensize(self):
        self.assertTrue(compare_lines_from_code(self.square_pensize_1_code, self.square_pensize_5_code, show_screen=False, pensize_invariant=True))
        self.assertFalse(compare_lines_from_code(self.square_pensize_1_code, self.square_pensize_5_code, show_screen=False, pensize_invariant=False))
    
    def test_same_shape_different_pencolor(self):
        self.assertFalse(compare_lines_from_code(self.square_pencolor_red_code, self.square_pencolor_blue_code, show_screen=False))
    
    def test_empty_code(self):
        self.assertFalse(compare_lines_from_code(self.empty_code, self.empty_code, show_screen=False))
    
    def test_empty_vs_non_empty_code(self):
        self.assertFalse(compare_lines_from_code(self.empty_code, self.square_100_code, show_screen=False))
    
    def test_discontinuous_vs_continuous_line(self):
        self.assertFalse(compare_lines_from_code(self.discontinuous_line_code, self.continuous_line_code, show_screen=False))
    
    def test_rotated_shapes(self):
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.square_rotated_45_code, show_screen=False))
    
    def test_negative_coordinates(self):
        self.assertTrue(compare_lines_from_code(self.square_100_code, self.square_negative_coords_code, show_screen=False, length_invariant=True, position_invariant=True))
        self.assertFalse(compare_lines_from_code(self.square_100_code, self.square_negative_coords_code, show_screen=False, length_invariant=True, position_invariant=False))
    
    def test_complex_shapes(self):
        self.assertTrue(compare_lines_from_code(self.star_code, self.star_alternative_code, show_screen=False))
    
    def test_circles(self):
        self.assertTrue(compare_lines_from_code(self.circle_100_code, self.circle_100_alternative_code, show_screen=False))
    
    def test_filled_shapes_different_positions(self):
        self.assertTrue(compare_lines_from_code(self.filled_square_red_code, self.filled_square_offset_code, show_screen=False, length_invariant=True, position_invariant=True))
        self.assertFalse(compare_lines_from_code(self.filled_square_red_code, self.filled_square_offset_code, show_screen=False, length_invariant=True, position_invariant=False))
    
    def test_overlapping_shapes_different_order(self):
        self.assertFalse(compare_lines_from_code(self.overlapping_shapes_1_code, self.overlapping_shapes_2_code, show_screen=False))
    
    def test_code_with_syntax_error(self):
        self.assertFalse(compare_lines_from_code(self.code_with_error_code, self.square_100_code, show_screen=False))
    
    def test_shapes_with_stamp(self):
        self.assertFalse(compare_lines_from_code(self.code_with_stamp_code, self.code_with_stamp_code, show_screen=False))
    
    def test_shapes_with_different_speeds(self):
        self.assertTrue(compare_lines_from_code(self.square_speed_1_code, self.square_speed_10_code, show_screen=False))
    
    def test_complex_penup_pendown_sequences(self):
        self.assertFalse(compare_lines_from_code(self.complex_penup_pendown_1_code, self.complex_penup_pendown_2_code, show_screen=False))
    
    def test_relative_vs_absolute_movement(self):
        self.assertTrue(compare_lines_from_code(self.relative_movement_code, self.absolute_movement_code, show_screen=False, position_invariant=True))
    
    def test_mirrored_shapes(self):
        self.assertTrue(compare_lines_from_code(self.square_100_code, self.square_mirrored_code, show_screen=False))
    
    def test_shapes_drawn_in_opposite_directions(self):
        self.assertTrue(compare_lines_from_code(self.square_counter_clockwise_code, self.square_clockwise_code, show_screen=False, length_invariant=True, position_invariant=True))
    
    def test_11g_methods(self):
        self.assertTrue(compare_lines_from_code(self.method_11g_code_1, self.method_11g_code_2, show_screen=False))
    
    def test_llava_output(self):
        self.assertFalse(compare_lines_from_code(self.llava_output_code, self.square_100_code, show_screen=False))
    
    def test_octagon(self):
        self.assertTrue(compare_lines_from_code(self.octagon_code_1, self.octagon_code_2, show_screen=False))
    
    def test_octagon_color(self):
        self.assertTrue(compare_lines_from_code(self.octagon_color_code_1, self.octagon_color_code_2, show_screen=False))
    
    def test_cross_11j(self):
        self.assertTrue(compare_lines_from_code(self.cross_11j_code_1, self.cross_11j_code_2, show_screen=False))
    
    def test_radio(self):
        self.assertTrue(compare_lines_from_code(self.radio_1, self.radio_2, show_screen=False))
    
    def test_sparse_radio(self):
        self.assertTrue(compare_lines_from_code(self.sparse_radio_1, self.sparse_radio_2, show_screen=False))
    
    def test_red_square_white_square(self):
        # TODO: this is not solved yet (corner case)
        self.assertFalse(compare_lines_from_code(self.code_red_square_white_square, self.code_red_square_white_square_2, show_screen=False))
    
    def test_midi_7d_chatcmpl_f8f375e5(self):
        self.assertFalse(compare_lines_from_code(self.code_midi_7d_chatcmpl_f8f375e5_output, self.code_midi_7d_chatcmpl_f8f375e5_true, show_screen=False))
    
    def test_midi_3j_chatcmpl_c2c70e38(self):
        self.assertFalse(compare_lines_from_code(self.code_midi_3j_chatcmpl_c2c70e38_output, self.code_midi_3j_chatcmpl_c2c70e38_true, show_screen=False))
    
    # TODO: there is no way to get this to pass, because it contains a fill color;
    # and the difference is only 0.32% different pixels
    # def test_midi_3j_chatcmpl_aa8e315c(self):
    #     self.assertFalse(compare_lines_from_code(self.code_midi_3j_chatcmpl_aa8e315c_output, self.code_midi_3j_chatcmpl_aa8e315c_true, show_screen=False))
        
    def test_midi_11j_chatcmpl_cb7f1d90(self):
        self.assertFalse(compare_lines_from_code(self.code_midi_11j_chatcmpl_cb7f1d90_output, self.code_midi_11j_chatcmpl_cb7f1d90_true, show_screen=False))
        
    
    def test_midi_10h_chatcmpl_32cde675(self):
        self.assertTrue(compare_lines_from_code(self.code_midi_10h_chatcmpl_32cde675_output, self.code_midi_10h_chatcmpl_32cde675_true, show_screen=False))
    
    def test_midi_8g_chatcmpl_f466813c(self):
        # TODO: this is not solved yet (this is the limitation of the symbolic evaluation for circles)
        self.assertTrue(compare_lines_from_code(self.code_midi_8g_chatcmpl_f466813c_output, self.code_midi_8g_chatcmpl_f466813c_true, show_screen=False))
    
    def test_midi_8d_chatcmpl_35e23aa7(self):
        # TODO: this is not solved yet (this is the limitation of the symbolic evaluation for circles)
        self.assertTrue(compare_lines_from_code(self.code_midi_8d_chatcmpl_35e23aa7_output, self.code_midi_8d_chatcmpl_35e23aa7_true, show_screen=False))
    
    def test_midi_7b_chatcmpl_4bc4a92b(self):
        self.assertTrue(compare_lines_from_code(self.code_midi_7b_chatcmpl_4bc4a92b_output, self.code_midi_7b_chatcmpl_4bc4a92b_true, show_screen=False))
    
    def test_midi_11j_chatcmpl_f34a22a4(self):
        self.assertTrue(compare_lines_from_code(self.code_midi_11j_chatcmpl_f34a22a4_output, self.code_midi_11j_chatcmpl_f34a22a4_true, show_screen=False))
    
    
if __name__ == '__main__':
    unittest.main()

    # suite = unittest.TestSuite()
    # suite.addTest(TestCompareLines('test_11g_methods'))
    # suite.addTest(TestCompareLines('test_llava_output'))
    # suite.addTest(TestCompareLines('test_overlapping_shapes_different_order'))
    # suite.addTest(TestCompareLines('test_filled_shape_with_outline'))
    # suite.addTest(TestCompareLines('test_same_shape_different_pensize'))
    # suite.addTest(TestCompareLines('test_octagon'))
    # suite.addTest(TestCompareLines('test_octagon_color'))
    # suite.addTest(TestCompareLines('test_empty_code'))
    # suite.addTest(TestCompareLines('test_same_lines_different_order'))
    # suite.addTest(TestCompareLines('test_empty_vs_non_empty_code'))
    # suite.addTest(TestCompareLines('test_shapes_with_stamp'))
    # suite.addTest(TestCompareLines('test_cross_11j'))
    # suite.addTest(TestCompareLines('test_same_shape_different_fillcolor'))
    # suite.addTest(TestCompareLines('test_radio'))
    # suite.addTest(TestCompareLines('test_sparse_radio'))
    # suite.addTest(TestCompareLines('test_complex_penup_pendown_sequences'))
    # suite.addTest(TestCompareLines('test_midi_11j_chatcmpl_cb7f1d90'))
    # suite.addTest(TestCompareLines('test_midi_7d_chatcmpl_f8f375e5'))
    # suite.addTest(TestCompareLines('test_midi_3j_chatcmpl_c2c70e38'))
    # suite.addTest(TestCompareLines('test_midi_10h_chatcmpl_32cde675'))
    # suite.addTest(TestCompareLines('test_red_square_white_square')) # TODO: not solved yet (corner case)
    # suite.addTest(TestCompareLines('test_midi_8g_chatcmpl_f466813c')) # TODO: circle-based, not solved
    # suite.addTest(TestCompareLines('test_midi_8d_chatcmpl_35e23aa7')) # TODO: circle-based, not solved
    # suite.addTest(TestCompareLines('test_midi_7b_chatcmpl_4bc4a92b'))
    # suite.addTest(TestCompareLines('test_midi_11j_chatcmpl_f34a22a4'))
    # unittest.TextTestRunner().run(suite)
