"""
Utility functions for normalizing turtle graphics lines.
"""


def get_bounding_box(lines, precision=6):
    """
    Get the bounding box coordinates and dimensions for a set of lines/shapes.
    Returns (min_x, max_x, min_y, max_y, width, height) rounded to specified precision.
    """
    if not lines:
        return None
    
    x_coords = []
    y_coords = []
    for shape in lines:
        if shape.get('type') == 'line':
            x_coords.extend([shape['start'][0], shape['end'][0]])
            y_coords.extend([shape['start'][1], shape['end'][1]])
        elif shape.get('type') == 'fill':
            for vertex in shape['vertices']:
                x_coords.append(vertex[0])
                y_coords.append(vertex[1])
            # Include lines inside fill shapes
            if 'lines' in shape:
                for line in shape['lines']:
                    x_coords.extend([line['start'][0], line['end'][0]])
                    y_coords.extend([line['start'][1], line['end'][1]])

    if not x_coords or not y_coords:
        return (0, 0, 0, 0, 0, 0)  # Return a default bounding box or handle the empty case

    min_x = round(min(x_coords), precision)
    max_x = round(max(x_coords), precision)
    min_y = round(min(y_coords), precision)
    max_y = round(max(y_coords), precision)
    width = round(max_x - min_x, precision)
    height = round(max_y - min_y, precision)

    return (min_x, max_x, min_y, max_y, width, height)

def normalize_length_of_lines(lines, target_max_length=300, precision=6):
    """
    Normalize all lines by scaling them based on the maximum dimension 
    of the entire pattern's bounding box.
    
    Args:
        lines: List of line/shape dictionaries
        target_max_length: Target maximum dimension (default 1000 for better precision)
        precision: Number of decimal places to round to (default 6)
    """
    if not lines:
        return []
    
    bbox = get_bounding_box(lines)
    if bbox is None:
        return lines.copy()
    
    _, _, _, _, width, height = bbox
    max_dimension = max(width, height)
    
    if max_dimension == 0:
        return lines.copy()
    
    # Calculate scaling factor
    scale_factor = target_max_length / max_dimension
    
    # Create normalized copy of lines
    normalized_states = []
    for shape in lines:
        if shape.get('type') == 'line':
            new_shape = shape.copy()
            new_shape['start'] = (
                round(shape['start'][0] * scale_factor, precision),
                round(shape['start'][1] * scale_factor, precision)
            )
            new_shape['end'] = (
                round(shape['end'][0] * scale_factor, precision),
                round(shape['end'][1] * scale_factor, precision)
            )
            normalized_states.append(new_shape)
        elif shape.get('type') == 'fill':
            new_shape = shape.copy()
            # Scale vertices
            new_vertices = []
            for vertex in shape['vertices']:
                new_vertex = (
                    round(vertex[0] * scale_factor, precision),
                    round(vertex[1] * scale_factor, precision)
                )
                new_vertices.append(new_vertex)
            new_shape['vertices'] = new_vertices
            
            # Scale lines inside fill if they exist
            if 'lines' in shape:
                new_lines = []
                for line in shape['lines']:
                    new_line = line.copy()
                    new_line['start'] = (
                        round(line['start'][0] * scale_factor, precision),
                        round(line['start'][1] * scale_factor, precision)
                    )
                    new_line['end'] = (
                        round(line['end'][0] * scale_factor, precision),
                        round(line['end'][1] * scale_factor, precision)
                    )
                    new_lines.append(new_line)
                new_shape['lines'] = new_lines
                
            normalized_states.append(new_shape)
    
    return normalized_states


def center_position_of_lines(lines, precision=6):
    """
    Center all lines around the origin by translating them.
    Then shift them so that all coordinates are non-negative.
    Rounds coordinates to specified precision to avoid floating-point comparison issues.
    """
    if not lines:
        return []
    
    bbox = get_bounding_box(lines)
    if bbox is None:
        return lines.copy()
    
    min_x, max_x, min_y, max_y, _, _ = bbox
    
    # Calculate center of the shape
    center_x = (min_x + max_x) / 2
    center_y = (min_y + max_y) / 2
    
    # First, center the lines
    shifted_lines = []
    for shape in lines:
        if shape.get('type') == 'line':
            new_shape = shape.copy()
            new_shape['start'] = (
                round(shape['start'][0] - center_x, precision),
                round(shape['start'][1] - center_y, precision)
            )
            new_shape['end'] = (
                round(shape['end'][0] - center_x, precision),
                round(shape['end'][1] - center_y, precision)
            )
            shifted_lines.append(new_shape)
        elif shape.get('type') == 'fill':
            new_shape = shape.copy()
            # Center vertices
            new_vertices = []
            for vertex in shape['vertices']:
                new_vertex = (
                    round(vertex[0] - center_x, precision),
                    round(vertex[1] - center_y, precision)
                )
                new_vertices.append(new_vertex)
            new_shape['vertices'] = new_vertices
            
            # Center lines inside fill if they exist
            if 'lines' in shape:
                new_lines = []
                for line in shape['lines']:
                    new_line = line.copy()
                    new_line['start'] = (
                        round(line['start'][0] - center_x, precision),
                        round(line['start'][1] - center_y, precision)
                    )
                    new_line['end'] = (
                        round(line['end'][0] - center_x, precision),
                        round(line['end'][1] - center_y, precision)
                    )
                    new_lines.append(new_line)
                new_shape['lines'] = new_lines
                
            shifted_lines.append(new_shape)
    
    # Now, compute the new bounding box after centering
    x_coords = []
    y_coords = []
    for shape in shifted_lines:
        if shape.get('type') == 'line':
            x_coords.extend([shape['start'][0], shape['end'][0]])
            y_coords.extend([shape['start'][1], shape['end'][1]])
        elif shape.get('type') == 'fill':
            for vertex in shape['vertices']:
                x_coords.append(vertex[0])
                y_coords.append(vertex[1])
            # Include lines inside fill
            if 'lines' in shape:
                for line in shape['lines']:
                    x_coords.extend([line['start'][0], line['end'][0]])
                    y_coords.extend([line['start'][1], line['end'][1]])
    
    min_x = min(x_coords)
    min_y = min(y_coords)
    
    # Shift coordinates so that they are all positive
    offset_x = -min_x if min_x < 0 else 0
    offset_y = -min_y if min_y < 0 else 0
    
    centered_states = []
    for shape in shifted_lines:
        if shape.get('type') == 'line':
            new_shape = shape.copy()
            new_shape['start'] = (
                shape['start'][0] + offset_x,
                shape['start'][1] + offset_y
            )
            new_shape['end'] = (
                shape['end'][0] + offset_x,
                shape['end'][1] + offset_y
            )
            centered_states.append(new_shape)
        elif shape.get('type') == 'fill':
            new_shape = shape.copy()
            # Shift vertices
            new_vertices = []
            for vertex in shape['vertices']:
                new_vertex = (
                    vertex[0] + offset_x,
                    vertex[1] + offset_y
                )
                new_vertices.append(new_vertex)
            new_shape['vertices'] = new_vertices
            
            # Shift lines inside fill if they exist
            if 'lines' in shape:
                new_lines = []
                for line in shape['lines']:
                    new_line = line.copy()
                    new_line['start'] = (
                        round(line['start'][0] + offset_x, precision),
                        round(line['start'][1] + offset_y, precision)
                    )
                    new_line['end'] = (
                        round(line['end'][0] + offset_x, precision),
                        round(line['end'][1] + offset_y, precision)
                    )
                    new_lines.append(new_line)
                new_shape['lines'] = new_lines
                
            centered_states.append(new_shape)
    
    return centered_states

def update_pensize_of_lines(lines, pensize=1):
    """
    Update the pensize of all lines and lines within fill shapes to the given value.
    """
    for shape in lines:
        if shape.get('type') == 'line':
            shape['pensize'] = pensize
        elif shape.get('type') == 'fill' and 'lines' in shape:
            for line in shape['lines']:
                line['pensize'] = pensize
    return lines

