import random
import math

def sample_seeds(seed_list, n_seeds_combine, n_combs_sample, ordered=True):
    """
    Efficiently sample unique permutations or combinations from the seed_list,
    handling unhashable seed elements by operating on their indices.

    Args:
    - seed_list (list): List of input seeds (can be any type, including unhashable types like dicts)
    - n_seeds_combine (int): Number of seeds to combine in each permutation/combination
    - n_combs_sample (int): Number of unique permutations/combinations to return
    - ordered (bool): Whether to consider the order (permutations) or not (combinations)

    Returns:
    A list of `n_combs_sample` unique permutations or combinations of the seed elements.
    """
    n_seeds = len(seed_list)
    if ordered:
        n_total_combs = math.perm(n_seeds, n_seeds_combine)
    else:
        n_total_combs = math.comb(n_seeds, n_seeds_combine)

    n_combs_sample = min(n_combs_sample, n_total_combs)
    sampled_indices_set = set()
    indices = list(range(n_seeds))

    while len(sampled_indices_set) < n_combs_sample:
        if ordered:
            perm_indices = tuple(random.sample(indices, n_seeds_combine))
        else:
            comb_indices = tuple(sorted(random.sample(indices, n_seeds_combine)))
            perm_indices = comb_indices
        sampled_indices_set.add(perm_indices)

    # Map indices back to the actual seed elements
    sampled_seed_perms = [
        tuple(seed_list[i] for i in indices_tuple)
        for indices_tuple in sampled_indices_set
    ]

    return sampled_seed_perms

# Test the corrected code
if __name__ == "__main__":
    seeds = [{'name': 'A'}, {'name': 'B'}, {'name': 'C'}, {'name': 'D'}]
    n_seeds_combine = 2
    n_combs_sample = 10
    print("Ordered permutations:")
    for perm in sample_seeds(seeds, n_seeds_combine, n_combs_sample, ordered=True):
        print(perm)
    print("\nUnordered combinations:")
    for comb in sample_seeds(seeds, n_seeds_combine, n_combs_sample, ordered=False):
        print(comb)
